/* =========================================================
   Co-Producer Add-on — Program Renderer UI v1
   - Pop-out program canvas into clean window for OBS capture
========================================================= */
(function(){
  'use strict';

  const CP = window.CP;
  if (!CP || !CP.dom || !CP.bus) return;
  if (!CP.program || !CP.program._internals) return;
  const S = CP.program._internals;

  // Inject small styles
  (function inject(){
    if (document.getElementById('cpProgramUIStyles')) return;
    const st = document.createElement('style');
    st.id = 'cpProgramUIStyles';
    st.textContent = `
      .cpPanel{font-family:system-ui,-apple-system,Segoe UI,Roboto,Arial; color:#eaeaea;}
      .cpPanel h3{margin:0 0 10px; font-size:14px; letter-spacing:.04em; opacity:.9;}
      .cpRow{display:flex; gap:10px; align-items:center; margin:10px 0;}
      .cpBtn{appearance:none; border:1px solid rgba(255,255,255,.18); background:rgba(0,0,0,.35); color:#fff; padding:10px 12px; border-radius:10px; cursor:pointer; font-weight:700;}
      .cpBtn:hover{border-color:rgba(255,255,255,.30); background:rgba(0,0,0,.45);}
      .cpBtn.primary{border-color:rgba(255,255,255,.28); background:rgba(255,59,48,.18);}
      .cpBtn.primary:hover{background:rgba(255,59,48,.28);}
      .cpMeta{font-size:12px; opacity:.8;}
      .cpCode{font-family:ui-monospace,SFMono-Regular,Menlo,Monaco,Consolas,monospace; font-size:12px; opacity:.9;}
      .cpSmall{font-size:12px; opacity:.75;}
      .cpPill{display:inline-block; padding:3px 8px; border-radius:999px; background:rgba(255,255,255,.10); border:1px solid rgba(255,255,255,.12);}
      .cpWarn{color:#ffd6d6;}
    `;
    document.head.appendChild(st);
  })();

  function getCanvas(){
    return document.getElementById('cpProgramCanvas');
  }

  function openPopout(){
    const canvas = getCanvas();
    if (!canvas) return;

    if (S.pop && !S.pop.closed){
      S.pop.focus();
      return;
    }

    // Create placeholder in main UI where the canvas was
    const parent = canvas.parentElement;
    if (!parent) return;

    const ph = document.createElement('div');
    ph.className = 'cpSmall';
    ph.style.cssText = 'display:flex;align-items:center;justify-content:center;width:100%;height:100%;background:#000;border-radius:12px;border:1px solid rgba(255,255,255,.10);';
    ph.textContent = 'PROGRAM popped out — close the pop-out window to return';
    S.popPlaceholder = ph;

    parent.replaceChild(ph, canvas);

    const w = window.open('about:blank', 'cp_program_popout', 'width=1280,height=720,menubar=no,toolbar=no,location=no,status=no,resizable=yes');
    if (!w){
      // restore if blocked
      parent.replaceChild(canvas, ph);
      S.popPlaceholder = null;
      return;
    }

    S.pop = w;

    // Build popout shell
    w.document.open();
    w.document.write(`<!doctype html>
<html><head><meta charset="utf-8"><meta name="viewport" content="width=device-width,initial-scale=1">
<title>ON AIR — Program</title>
<style>
html,body{margin:0;height:100%;background:#000;overflow:hidden;}
.wrap{position:fixed;inset:0;display:flex;align-items:center;justify-content:center;}
.canvas{width:100%;height:100%;display:block;}
.badge{position:fixed;left:14px;top:14px;font-family:system-ui,-apple-system,Segoe UI,Roboto,Arial;color:#fff;
  background:rgba(0,0,0,.45);border:1px solid rgba(255,255,255,.16);padding:6px 10px;border-radius:999px;font-weight:800;letter-spacing:.06em;}
</style></head><body>
<div class="wrap"></div>
<div class="badge">ON AIR</div>
</body></html>`);
    w.document.close();

    const wrap = w.document.querySelector('.wrap');
    canvas.classList.add('canvas');
    wrap.appendChild(canvas);

    // On close, move canvas back
    const restore = ()=>{
      try{
        if (!S.popPlaceholder) return;
        const c = getCanvas() || canvas; // canvas moved
        if (!c) return;
        const mainOnAir = document.getElementById('onAirScreen');
        if (!mainOnAir) return;

        mainOnAir.innerHTML = '';
        mainOnAir.appendChild(c);
        S.popPlaceholder = null;
      }catch(_){}
    };

    w.addEventListener('beforeunload', restore, { once:true });
  }

  function panel(){
    const st = CP.program.getState();
    return `
      <div class="cpPanel">
        <h3>PROGRAM RENDERER</h3>
        <div class="cpMeta">Output: <span class="cpPill">#onAirScreen</span> (canvas)</div>
        <div class="cpRow">
          <button class="cpBtn primary" id="cpPR_take">TAKE</button>
          <button class="cpBtn" id="cpPR_pop">POP-OUT</button>
        </div>
        <div class="cpRow">
          <div class="cpMeta">Transition: <span class="cpCode">${(st.transition&&st.transition.kind)||'cut'} / ${(st.transition&&st.transition.fadeMs)||0}ms</span></div>
        </div>
        <div class="cpRow">
          <div class="cpMeta">Program: <span class="cpCode">${st.program ? st.program.label : '—'}</span></div>
        </div>
        <div class="cpSmall">Tip: Use OBS “Window Capture” on this pop-out for clean streaming.</div>
      </div>
    `;
  }

  function openPanel(){
    const m = CP.dom.createModal(panel(), { title: 'Program Renderer', width: 420 });
    const root = m.root;
    root.querySelector('#cpPR_take')?.addEventListener('click', ()=> CP.bus.emit('program:take', { ui:true, at:Date.now() }));
    root.querySelector('#cpPR_pop')?.addEventListener('click', openPopout);

    // Update small live info every 500ms
    const t = setInterval(()=>{
      try{
        if (!document.body.contains(root)) { clearInterval(t); return; }
        root.innerHTML = panel();
        root.querySelector('#cpPR_take')?.addEventListener('click', ()=> CP.bus.emit('program:take', { ui:true, at:Date.now() }));
        root.querySelector('#cpPR_pop')?.addEventListener('click', openPopout);
      }catch(_){}
    }, 500);
  }

  CP.dom.addDockButton('Program', 'PR', openPanel);

})();