/* =========================================================
   Co-Producer Addons — PROGRAM PUBLISH UI
========================================================= */
(function(){
  "use strict";

  function el(tag, attrs={}, children=[]){
    const e = document.createElement(tag);
    for (const [k,v] of Object.entries(attrs||{})){
      if (k === "class") e.className = v;
      else if (k === "html") e.innerHTML = v;
      else if (k.startsWith("on") && typeof v === "function") e.addEventListener(k.slice(2), v);
      else e.setAttribute(k, String(v));
    }
    for (const c of (children||[])){
      if (c === null || c === undefined) continue;
      e.appendChild(typeof c === "string" ? document.createTextNode(c) : c);
    }
    return e;
  }

  function css(){
    return `
      .pp-row{display:flex; gap:10px; align-items:center; justify-content:space-between; padding:10px 0; border-bottom:1px solid rgba(255,255,255,.08)}
      .pp-row:last-child{border-bottom:none}
      .pp-k{color:rgba(255,255,255,.72); font-weight:800; font-size:12px; text-transform:uppercase; letter-spacing:.2px}
      .pp-v{color:rgba(255,255,255,.92); font-weight:900; font-size:13px}
      .pp-bad{display:inline-flex; align-items:center; gap:8px; padding:6px 10px; border-radius:999px; border:1px solid rgba(255,255,255,.12); background:rgba(0,0,0,.25); font-weight:900; font-size:12px}
      .pp-dot{width:10px;height:10px;border-radius:99px;background:#445; box-shadow: inset 0 -2px 4px rgba(0,0,0,.5)}
      .pp-dot.ok{background:#20c997; box-shadow:0 0 0 4px rgba(32,201,151,.12), 0 0 14px rgba(32,201,151,.55)}
      .pp-dot.off{background:#556}
      .pp-dot.live{background:#ff4a6a; box-shadow:0 0 0 4px rgba(255,74,106,.12), 0 0 14px rgba(255,74,106,.55); animation:ppflash .95s ease-in-out infinite}
      @keyframes ppflash{0%,100%{filter:brightness(1)}50%{filter:brightness(1.55)}}
      .pp-in{width:100%; padding:10px 10px; border-radius:12px; border:1px solid rgba(255,255,255,.12); background:rgba(0,0,0,.22); color:#eef3ff; font-weight:900; outline:none}
      .pp-in:focus{border-color:rgba(74,163,255,.55); box-shadow:0 0 0 4px rgba(74,163,255,.12)}
      .pp-btns{display:flex; gap:10px; flex-wrap:wrap; margin-top:12px}
      .pp-btn{border:1px solid rgba(255,255,255,.14); background:rgba(255,255,255,.06); color:#eef3ff; padding:10px 12px; border-radius:12px; cursor:pointer; font-weight:900}
      .pp-btn:hover{border-color:rgba(255,255,255,.22); background:rgba(255,255,255,.08)}
      .pp-btn.primary{border-color:rgba(74,163,255,.45); background:rgba(74,163,255,.18)}
      .pp-btn.danger{border-color:rgba(255,74,106,.45); background:rgba(255,74,106,.16)}
      .pp-btn.ok{border-color:rgba(32,201,151,.45); background:rgba(32,201,151,.14)}
      .pp-btn:disabled{opacity:.45; cursor:not-allowed}
      .pp-link{display:flex; gap:10px; align-items:center; flex-wrap:wrap}
      .pp-url{flex:1 1 auto; min-width:220px; padding:10px 10px; border-radius:12px; border:1px solid rgba(255,255,255,.12); background:rgba(0,0,0,.22); font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono","Courier New", monospace; font-size:12px; overflow:hidden; text-overflow:ellipsis; white-space:nowrap}
    `;
  }

  function getAPI(){
    return window.CPProgramPublish || null;
  }

  function fmtTime(ts){
    if (!ts) return "—";
    const d = new Date(ts);
    const hh = String(d.getHours()).padStart(2,"0");
    const mm = String(d.getMinutes()).padStart(2,"0");
    const ss = String(d.getSeconds()).padStart(2,"0");
    return `${hh}:${mm}:${ss}`;
  }

  function copy(txt){
    if (!txt) return;
    navigator.clipboard.writeText(txt).catch(() => {
      const ta = document.createElement("textarea");
      ta.value = txt;
      document.body.appendChild(ta);
      ta.select();
      try{ document.execCommand("copy"); }catch(_){}
      ta.remove();
    });
  }

  function mountUI(){
    const CP = window.CP;
    const api = getAPI();
    if (!CP || !api) return false;

    const style = el("style", { html: css() });

    const livekitBadge = el("div", { class:"pp-bad" }, [
      el("span", { class:"pp-dot off", id:"ppRoomDot" }),
      el("span", { class:"pp-v", id:"ppRoomText" }, ["Room: —"])
    ]);

    const canvasBadge = el("div", { class:"pp-bad" }, [
      el("span", { class:"pp-dot off", id:"ppCanvasDot" }),
      el("span", { class:"pp-v", id:"ppCanvasText" }, ["Canvas: —"])
    ]);

    const outBadge = el("div", { class:"pp-bad" }, [
      el("span", { class:"pp-dot off", id:"ppOutDot" }),
      el("span", { class:"pp-v", id:"ppOutText" }, ["Output: OFF"])
    ]);

    const trackName = el("input", { class:"pp-in", id:"ppTrackName", placeholder:"program" });
    const fps = el("select", { class:"pp-in", id:"ppFps" }, [
      el("option", { value:"15" }, ["15 fps"]),
      el("option", { value:"30" }, ["30 fps"]),
      el("option", { value:"60" }, ["60 fps"])
    ]);

    const btnToggle = el("button", { class:"pp-btn primary", id:"ppToggle" }, ["Publish PROGRAM"]);
    const btnStop = el("button", { class:"pp-btn danger", id:"ppStop" }, ["Stop"]);
    const btnOpen = el("button", { class:"pp-btn ok", id:"ppOpen" }, ["Open OnAir"]);
    const btnCopy = el("button", { class:"pp-btn", id:"ppCopy" }, ["Copy Link"]);

    const urlBox = el("div", { class:"pp-url", id:"ppUrl" }, ["—"]);

    const last = el("div", { class:"pp-row" }, [
      el("div", { class:"pp-k" }, ["Last publish"]),
      el("div", { class:"pp-v", id:"ppLast" }, ["—"])
    ]);

    const err = el("div", { class:"pp-row" }, [
      el("div", { class:"pp-k" }, ["Error"]),
      el("div", { class:"pp-v", id:"ppErr", style:"color: rgba(255,208,80,.95)" }, ["—"])
    ]);

    const body = el("div", {}, [
      style,
      el("div", { style:"display:flex; gap:10px; flex-wrap:wrap; margin-bottom:10px" }, [ livekitBadge, canvasBadge, outBadge ]),
      el("div", { class:"pp-row" }, [
        el("div", { class:"pp-k" }, ["Track name"]),
        el("div", { style:"flex:1 1 auto; max-width:280px" }, [ trackName ])
      ]),
      el("div", { class:"pp-row" }, [
        el("div", { class:"pp-k" }, ["FPS"]),
        el("div", { style:"flex:1 1 auto; max-width:280px" }, [ fps ])
      ]),
      el("div", { class:"pp-btns" }, [ btnToggle, btnStop ]),
      el("div", { style:"height:10px" }),
      el("div", { class:"pp-link" }, [ urlBox, btnOpen, btnCopy ]),
      last,
      err
    ]);

    const modal = CP.dom.createModal({
      title: "PROGRAM OUT",
      content: body,
      actions: [{ label:"Close", onClick: () => modal.close() }]
    });

    function setDot(dotEl, mode){
      dotEl.classList.remove("ok","off","live");
      if (mode === "ok") dotEl.classList.add("ok");
      else if (mode === "live") dotEl.classList.add("live");
      else dotEl.classList.add("off");
    }

    function refresh(){
      const st = api.getState();

      const room = (window.CP_CTX && window.CP_CTX.getRoom && window.CP_CTX.getRoom()) || null;
      const roomName = api.currentRoomName() || (room && room.name) || "—";
      const roomDot = document.getElementById("ppRoomDot");
      const roomText = document.getElementById("ppRoomText");
      if (roomText) roomText.textContent = `Room: ${roomName || "—"}`;
      if (roomDot) setDot(roomDot, room ? "ok" : "off");

      const canvasDot = document.getElementById("ppCanvasDot");
      const canvasText = document.getElementById("ppCanvasText");
      if (canvasText) canvasText.textContent = st.hasCanvas ? "Canvas: OK" : "Canvas: missing";
      if (canvasDot) setDot(canvasDot, st.hasCanvas ? "ok" : "off");

      const outDot = document.getElementById("ppOutDot");
      const outText = document.getElementById("ppOutText");
      if (outText) outText.textContent = st.published ? "Output: LIVE" : "Output: OFF";
      if (outDot) setDot(outDot, st.published ? "live" : "off");

      const lastEl = document.getElementById("ppLast");
      if (lastEl) lastEl.textContent = st.lastPublishedAt ? fmtTime(st.lastPublishedAt) : "—";

      const errEl = document.getElementById("ppErr");
      if (errEl) errEl.textContent = st.lastError ? st.lastError : "—";

      const urlEl = document.getElementById("ppUrl");
      const u = api.getOnAirUrl();
      if (urlEl) urlEl.textContent = u || "—";

      btnToggle.textContent = st.published ? "Unpublish PROGRAM" : "Publish PROGRAM";
      btnToggle.classList.toggle("danger", st.published);
      btnToggle.classList.toggle("primary", !st.published);

      btnStop.disabled = !st.published;
      btnOpen.disabled = !u;
      btnCopy.disabled = !u;
    }

    // Initialize inputs from state
    const st0 = api.getState();
    trackName.value = st0.trackName || "program";
    fps.value = String(st0.fps || 30);

    trackName.addEventListener("input", () => api.setTrackName(trackName.value));
    fps.addEventListener("change", () => api.setFps(fps.value));

    btnToggle.addEventListener("click", async () => {
      btnToggle.disabled = true;
      try{
        await api.toggle();
      }catch(e){
        try{ api.setError(String(e && (e.message || e) || e)); }catch(_){ }
        console.error(e);
      }finally{
        btnToggle.disabled = false;
        refresh();
      }
    });

    btnStop.addEventListener("click", async () => {
      btnStop.disabled = true;
      try{ await api.stop(); }catch(e){ console.error(e); }finally{ btnStop.disabled = false; refresh(); }
    });

    btnOpen.addEventListener("click", () => {
      const u = api.getOnAirUrl();
      if (u) window.open(u, "_blank", "noopener,noreferrer");
    });

    btnCopy.addEventListener("click", () => {
      const u = api.getOnAirUrl();
      if (u){ copy(u); btnCopy.textContent = "Copied"; setTimeout(()=>btnCopy.textContent="Copy Link", 900); }
    });

    // Listen changes
    api.onChange(refresh);

    // periodic refresh (room connects later)
    const iv = setInterval(refresh, 600);
    modal.onClose(() => clearInterval(iv));

    refresh();
    return true;
  }

  if (window.CP && typeof window.CP.registerPlugin === "function"){
    window.CP.registerPlugin({
      id: "program_publish_ui",
      name: "Program Publish UI",
      init(){
        // Add to dock
        const CP = window.CP;
        CP.dom.addDockButton("📡", "Program Out", mountUI);
      }
    });
  } else {
    // If plugin host isn't ready, retry
    let tries = 0;
    const iv = setInterval(() => {
      tries++;
      if (mountUI() || tries > 50) clearInterval(iv);
    }, 200);
  }
})();
