/* =========================================================
   Co-Producer Addons — PROGRAM PUBLISH (push PROGRAM canvas to LiveKit)
   - Publishes #cpProgramCanvas as a LiveKit video track named "program"
   - Does NOT require edits to coproducer_livekit.html
   - Depends on: core_bridge (CP_CTX.getRoom)
========================================================= */
(function(){
  "use strict";

  const STORE = {
    key: "cp_program_publish_v1",
    load(){
      try{ return JSON.parse(localStorage.getItem(this.key) || "{}") || {}; }catch(_){ return {}; }
    },
    save(obj){
      try{ localStorage.setItem(this.key, JSON.stringify(obj||{})); }catch(_){}
    }
  };

  function getLK(){
    return (window.CP_CTX && window.CP_CTX.getLK && window.CP_CTX.getLK()) ||
      window.LivekitClient || window.LK || window.livekitClient || window.livekit || null;
  }

  function getRoom(){
    return (window.CP_CTX && window.CP_CTX.getRoom && window.CP_CTX.getRoom()) || window.__CP_LK_ROOM__ || null;
  }

  function pickProgramCanvas(){
    // Primary: our renderer plugin canvas
    const a = document.getElementById("cpProgramCanvas");
    if (a && typeof a.captureStream === "function") return a;

    // Fallbacks (best-effort): on-air canvas/video
    const b = document.getElementById("onAirScreen");
    if (b && typeof b.captureStream === "function") return b;

    const c = document.querySelector("canvas");
    if (c && typeof c.captureStream === "function") return c;

    return null;
  }

  async function unpublishExistingByName(room, name){
    try{
      const lp = room && room.localParticipant;
      if (!lp) return;

      // videoTrackPublications is a Map (sid -> publication)
      const pubs = lp.videoTrackPublications ? Array.from(lp.videoTrackPublications.values()) : [];
      for (const pub of pubs){
        const tn = pub && (pub.trackName || (pub.track && pub.track.name));
        if (tn === name){
          try{
            // unpublishTrack expects LocalTrack; publication.track should be LocalTrack
            if (pub.track) lp.unpublishTrack(pub.track);
          }catch(_){}
        }
      }
    }catch(_){}
  }

  const listeners = new Set();
  function emit(){
    for (const fn of listeners){
      try{ fn(getState()); }catch(_){}
    }
  }
  function onChange(fn){
    listeners.add(fn);
    return () => listeners.delete(fn);
  }

  const state = {
    enabled: true,
    published: false,
    trackName: "program",
    fps: 30,
    lastError: "",
    lastPublishedAt: 0
  };

  const pub = {
    stream: null,
    mst: null,
    publication: null,
    roomSid: ""
  };

  function load(){
    const s = STORE.load();
    if (typeof s.trackName === "string" && s.trackName.trim()) state.trackName = s.trackName.trim();
    if (Number.isFinite(s.fps)) state.fps = Math.max(5, Math.min(60, s.fps));
    state.enabled = (typeof s.enabled === "boolean") ? s.enabled : true;
    emit();
  }

  function save(){
    STORE.save({ enabled: state.enabled, trackName: state.trackName, fps: state.fps });
  }

  function getState(){
    return {
      enabled: state.enabled,
      published: state.published,
      trackName: state.trackName,
      fps: state.fps,
      lastError: state.lastError,
      lastPublishedAt: state.lastPublishedAt,
      hasCanvas: !!pickProgramCanvas(),
      hasRoom: !!getRoom()
    };
  }

  function setTrackName(name){
    state.trackName = String(name || "").trim() || "program";
    save();
    emit();
  }

  function setFps(fps){
    const n = parseInt(String(fps), 10);
    if (!Number.isFinite(n)) return;
    state.fps = Math.max(5, Math.min(60, n));
    save();
    emit();
  }

  function setEnabled(v){
    state.enabled = !!v;
    save();
    emit();
  }

  async function publish(){
    state.lastError = "";
    emit();

    const room = getRoom();
    if (!room) throw new Error("LiveKit room not detected yet. Join room first.");

    const canvas = pickProgramCanvas();
    if (!canvas) throw new Error("Program canvas not found. Ensure Program Renderer plugin is enabled.");

    // Create capture stream
    const fps = state.fps || 30;
    const stream = canvas.captureStream(fps);
    const mst = stream.getVideoTracks()[0];
    if (!mst) throw new Error("No video track from canvas.captureStream().");

    // In case we re-publish, stop previous
    await stop();

    pub.stream = stream;
    pub.mst = mst;

    // Publish (JS SDK supports MediaStreamTrack directly)
    const LK = getLK();
    const lp = room.localParticipant;
    if (!lp || typeof lp.publishTrack !== "function") throw new Error("LocalParticipant.publishTrack not available.");

    await unpublishExistingByName(room, state.trackName);

    const opts = { name: state.trackName };
    try{
      if (LK && LK.Track && LK.Track.Source && LK.Track.Source.ScreenShare) opts.source = LK.Track.Source.ScreenShare;
    }catch(_){}

    const publication = await lp.publishTrack(mst, opts);
    pub.publication = publication;
    pub.roomSid = room.sid || "";

    state.published = true;
    state.lastPublishedAt = Date.now();
    emit();

    // Keep the track alive
    mst.onended = () => { stop().catch(()=>{}); };

    return publication;
  }

  async function stop(){
    const room = getRoom();
    const lp = room && room.localParticipant;

    // Unpublish first (best-effort)
    try{
      if (lp && pub.publication && pub.publication.track){
        lp.unpublishTrack(pub.publication.track);
      }
    }catch(_){}

    try{
      if (pub.mst) pub.mst.stop();
    }catch(_){}

    try{
      if (pub.stream) pub.stream.getTracks().forEach(t => { try{ t.stop(); }catch(_){} });
    }catch(_){}

    pub.stream = null;
    pub.mst = null;
    pub.publication = null;
    pub.roomSid = "";

    if (state.published){
      state.published = false;
      emit();
    }
  }

  function setError(msg){
    state.lastError = String(msg || "");
    emit();
  }

  async function toggle(){
    if (state.published) return stop();
    return publish();
  }

  function buildOnAirUrl(roomName){
    const base = new URL("onair_program.html", location.href);
    if (roomName) base.searchParams.set("room", roomName);
    // We keep defaults for lk/token in the page itself; can override via ?lk=&token=
    return base.toString();
  }

  function currentRoomName(){
    const room = getRoom();
    // try a few known properties
    return (room && (room.name || room.roomName || room.options && room.options.name)) || "";
  }

  function getOnAirUrl(){
    return buildOnAirUrl(currentRoomName());
  }

  // Expose API for UI plugin
  window.CPProgramPublish = {
    load,
    save,
    onChange,
    getState,
    setTrackName,
    setFps,
    setEnabled,
    setError,
    publish,
    stop,
    toggle,
    getOnAirUrl,
    currentRoomName
  };

  // Auto-load saved state
  load();

  // Register plugin (logic only)
  if (window.CP && typeof window.CP.registerPlugin === "function"){
    window.CP.registerPlugin({
      id: "program_publish",
      name: "Program Publish",
      init(){
        // nothing else; UI plugin drives it
      }
    });
  }
})();
