/* =========================================================
   Virtuals Plugin — UI (v1)
   File: addons/virtuals.ui.js
========================================================= */
(function(){
  "use strict";

  function ensureStyles(){
    if (document.getElementById("cpVirtualsStyles")) return;
    const css = `
      .viWrap{ padding:14px; }
      .viTop{ display:flex; align-items:center; justify-content:space-between; gap:12px; flex-wrap:wrap; margin-bottom:12px; }
      .viPill{ display:inline-flex; align-items:center; gap:8px; padding:6px 10px; border-radius:999px;
        border:1px solid rgba(255,255,255,.12); background:rgba(0,0,0,.18); font-size:12px; font-weight:900; }
      .viPill b{ font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono","Courier New", monospace; }
      .viCard{ border:1px solid rgba(255,255,255,.10); border-radius:14px; background:rgba(0,0,0,.12); padding:12px; }
      .viRow{ display:grid; grid-template-columns: 140px 1fr; gap:10px 12px; align-items:center; }
      .viLbl{ color:rgba(255,255,255,.74); font-size:12px; }
      .viSel, .viInp{ width:100%; padding:10px 10px; border-radius:12px; border:1px solid rgba(255,255,255,.12);
        background:rgba(0,0,0,.22); color:rgba(255,255,255,.92); outline:none; }
      .viSel:focus, .viInp:focus{ border-color: rgba(74,163,255,.55); box-shadow:0 0 0 4px rgba(74,163,255,.12); }
      .viBtns{ display:flex; gap:10px; justify-content:flex-end; flex-wrap:wrap; margin-top:10px; }
      .viBtn{ border:1px solid rgba(255,255,255,.14); background:rgba(255,255,255,.08); color:rgba(255,255,255,.92);
        padding:10px 12px; border-radius:12px; cursor:pointer; font-weight:1000; }
      .viBtn.primary{ border-color: rgba(74,163,255,.45); background: rgba(74,163,255,.14); }
      .viBtn:hover{ filter:brightness(1.08); }
      .viPrev{ margin-top:10px; display:flex; gap:12px; align-items:flex-start; }
      .viImgBox{ width:200px; height:112px; border-radius:12px; border:1px solid rgba(255,255,255,.10);
        background:rgba(0,0,0,.22); overflow:hidden; display:flex; align-items:center; justify-content:center; }
      .viImgBox img{ width:100%; height:100%; object-fit:cover; display:block; }
      .viHelp{ color:rgba(255,255,255,.66); font-size:12px; line-height:1.35; }
      .viPre{ margin-top:10px; padding:10px; border-radius:12px; border:1px solid rgba(255,255,255,.10);
        background:rgba(0,0,0,.22); font-size:12px; max-height:220px; overflow:auto; }
    `;
    const st = document.createElement("style");
    st.id = "cpVirtualsStyles";
    st.textContent = css;
    document.head.appendChild(st);
  }

  function findRendererFrames(){
    const frames = [];
    const byId = ["programRenderer","programFrame","rendererFrame","previewFrame","programPreview"];
    for (const id of byId){
      const el = document.getElementById(id);
      if (el && el.tagName === "IFRAME") frames.push(el);
    }
    document.querySelectorAll("iframe[data-program-renderer], iframe[data-renderer], iframe").forEach(f=>{
      if (f && !frames.includes(f)) frames.push(f);
    });
    return frames;
  }

  function postToRenderer(type, payload){
    const frames = findRendererFrames();
    for (const f of frames){
      try{ f.contentWindow && f.contentWindow.postMessage({ type, payload }, "*"); }catch(_){}
    }
    return frames.length;
  }

  function init(CP){
    if (!window.CPVirtuals){
      console.warn("[virtuals] CPVirtuals missing (logic not loaded?)");
      return;
    }
    ensureStyles();

    const api = window.CPVirtuals;
    const modal = CP.dom.createModal({ id:"cp_virtuals_modal", title:"Virtuals", pillText:"READY" });

    function build(){
      const cfg = api.load();

      modal.body.innerHTML = `
        <div class="viWrap">
          <div class="viTop">
            <div class="viPill">Mode: <b id="viMode">${cfg.mode}</b></div>
            <div class="viPill">Target: <b id="viTarget">${cfg.target}</b></div>
          </div>

          <div class="viCard">
            <div class="viRow">
              <div class="viLbl">Target</div>
              <select class="viSel" id="viT">
                <option value="local">local</option>
                <option value="remote">remote</option>
                <option value="program">program</option>
              </select>

              <div class="viLbl">Mode</div>
              <select class="viSel" id="viM">
                <option value="none">none</option>
                <option value="blur">blur</option>
                <option value="image">image</option>
              </select>

              <div class="viLbl">Blur</div>
              <input class="viInp" id="viBlur" placeholder="8" />

              <div class="viLbl">Image</div>
              <input class="viInp" id="viFileName" placeholder="choose file…" readonly />
              <div class="viLbl"></div>
              <input class="viInp" id="viFile" type="file" accept="image/*" />
            </div>

            <div class="viPrev">
              <div class="viImgBox" id="viImgBox"><div class="viHelp">No image</div></div>
              <div class="viHelp">
                Virtuals are applied by your <b>Program renderer</b>.
                This plugin only sends state via <b>postMessage</b>:
                <div style="margin-top:6px;font-family:ui-monospace">cp:virtuals:set</div>
              </div>
            </div>

            <div class="viBtns">
              <button class="viBtn primary" id="viApply" type="button">APPLY TO PREVIEW</button>
              <button class="viBtn" id="viSave" type="button">SAVE</button>
              <button class="viBtn" id="viReset" type="button">RESET</button>
            </div>

            <pre class="viPre" id="viPayload">{}</pre>
          </div>
        </div>
      `;

      const tEl = document.getElementById("viT");
      const mEl = document.getElementById("viM");
      const bEl = document.getElementById("viBlur");
      const fEl = document.getElementById("viFile");
      const fnEl = document.getElementById("viFileName");
      const imgBox = document.getElementById("viImgBox");

      tEl.value = cfg.target || "local";
      mEl.value = cfg.mode || "none";
      bEl.value = String(cfg.blur || 8);

      function renderImg(dataUrl){
        imgBox.innerHTML = "";
        if (!dataUrl){
          const d = document.createElement("div");
          d.className = "viHelp";
          d.textContent = "No image";
          imgBox.appendChild(d);
          return;
        }
        const img = document.createElement("img");
        img.src = dataUrl;
        imgBox.appendChild(img);
      }
      renderImg(cfg.imageDataUrl || "");

      function updatePayload(){
        const cur = api.save({
          target: tEl.value,
          mode: mEl.value,
          blur: parseInt(bEl.value,10) || 0
        });
        document.getElementById("viMode").textContent = cur.mode;
        document.getElementById("viTarget").textContent = cur.target;
        const p = api.payload();
        document.getElementById("viPayload").textContent = JSON.stringify(p, null, 2);
        return p;
      }

      async function apply(){
        const p = updatePayload();
        const frames = postToRenderer("cp:virtuals:set", p);
        CP.bus.emit("virtuals:apply", { payload: p, frames });
        modal.setPill(frames ? "SENT" : "NO RENDERER");
      }

      function save(){ updatePayload(); modal.setPill("SAVED"); }
      function reset(){
        if (!confirm("Reset virtuals?")) return;
        api.save({ mode:"none", blur:8, imageDataUrl:"", target:"local" });
        build();
      }

      fEl.addEventListener("change", ()=>{
        const file = fEl.files && fEl.files[0];
        if (!file) return;
        fnEl.value = file.name || "image";
        const reader = new FileReader();
        reader.onload = ()=>{
          const cfg2 = api.save(Object.assign(api.load(), { imageDataUrl: String(reader.result || "") }));
          renderImg(cfg2.imageDataUrl || "");
          updatePayload();
        };
        reader.readAsDataURL(file);
      });

      tEl.addEventListener("change", updatePayload);
      mEl.addEventListener("change", updatePayload);
      bEl.addEventListener("input", updatePayload);

      document.getElementById("viApply").addEventListener("click", apply);
      document.getElementById("viSave").addEventListener("click", save);
      document.getElementById("viReset").addEventListener("click", reset);

      updatePayload();
    }

    function open(){ build(); modal.open(); }

    const hook = CP.hookClick("#qaVirtuals", open, "replace");
    if (!hook.ok) CP.dom.addDockButton({ id:"cpDockVirtuals", icon:"🟦", title:"Virtuals", onClick: open });

    window.CPVirtualsUI = { open };
  }

  if (window.CP && typeof window.CP.registerPlugin === "function"){
    window.CP.registerPlugin({ id:"virtuals", init });
  } else {
    console.warn("[virtuals] CP plugin host not found. Include addons/plugin_host.js first.");
  }
})();
