/* =========================================================
   Co-Producer Add-on — Program Renderer (Canvas) v1
   - No core page edits required (hooks via globals)
   - Consumes CP.bus events from overlays/transitions/virtuals UI
   - Renders PROGRAM output into #onAirScreen (replaces RETURN view)
   - Optional: Pop-out canvas to a clean window for OBS capture
========================================================= */
(function(){
  'use strict';

  const CP = window.CP;
  if (!CP || !CP.bus){
    console.warn('[program_renderer] CP plugin host not found; abort.');
    return;
  }
  if (window.__CP_PROGRAM_RENDERER_V1__) return;
  window.__CP_PROGRAM_RENDERER_V1__ = true;

  const LK = window.LK || window.LiveKitClient || window.livekit || null;

  const state = {
    enabled: true,

    // incoming settings
    transition: { kind: 'cut', fadeMs: 250, stingerUrl: '', version: 1, at: Date.now() },
    overlays: [],
    virtuals: { mode: 'none', blur: 12, imageDataUrl: '', target: 'program' },

    // program routing
    programSource: null,    // resolved
    pendingSource: null,    // resolved
    transFrom: null,        // resolved
    transTo: null,          // resolved
    transKind: 'cut',
    transStart: 0,
    transDur: 0,

    // drawing
    w: 1280,
    h: 720,
    lastTs: 0,
    tickerX: 0,

    // media caches
    hiddenWrap: null,
    lkVideoEls: new Map(),     // key -> <video>
    mediaVideoEls: new Map(),  // id -> <video>
    mediaImages: new Map(),    // id -> <img>
    bgImg: null,

    // output
    canvas: null,
    ctx: null,

    // pop-out
    pop: null,
    popPlaceholder: null,

    // publish (future)
    publishEnabled: false,
    published: { track: null, sender: null }
  };

  function clamp(n, a, b){ return Math.max(a, Math.min(b, n)); }
  function now(){ return Date.now(); }

  function ensureHiddenWrap(){
    if (state.hiddenWrap) return state.hiddenWrap;
    const d = document.createElement('div');
    d.id = 'cpProgramHiddenMedia';
    d.style.cssText = 'position:fixed;left:-99999px;top:-99999px;width:1px;height:1px;overflow:hidden;opacity:0;pointer-events:none;';
    document.body.appendChild(d);
    state.hiddenWrap = d;
    return d;
  }

  function ensureCanvas(){
    if (state.canvas && state.ctx) return true;

    const onAirScreen = document.getElementById('onAirScreen');
    if (!onAirScreen) return false;

    // Take ownership of OnAir screen (avoid core updateOnAirPanel wiping our output)
    onAirScreen.innerHTML = '';
    onAirScreen.style.position = 'relative';

    const c = document.createElement('canvas');
    c.id = 'cpProgramCanvas';
    c.width = state.w;
    c.height = state.h;
    c.style.cssText = [
      'width:100%',
      'height:100%',
      'display:block',
      'background:#000',
      'border-radius:12px'
    ].join(';');

    onAirScreen.appendChild(c);

    state.canvas = c;
    state.ctx = c.getContext('2d');

    // Override the core on-air renderer so it doesn't fight the program canvas
    if (typeof window.updateOnAirPanel === 'function'){
      if (!window.__cp_orig_updateOnAirPanel) window.__cp_orig_updateOnAirPanel = window.updateOnAirPanel;
      window.updateOnAirPanel = function(){ /* program renderer owns onAirScreen */ };
    }

    // Seed ticker position
    state.tickerX = state.w;

    // Start render loop
    state.lastTs = 0;
    requestAnimationFrame(drawLoop);

    return true;
  }

  function safeUid(u){
    try{ return (window.safeUid ? window.safeUid(u) : (u||'')).replace(/[^a-z0-9_\-]/gi,'_'); }catch(_){ return 'u'; }
  }

  function resolveSource(src){
    if (!src) return null;

    // Preview source from core typically: { type:'slot', uid:'...' } OR { type:'media', id:'...' }
    const t = src.type || src.kind || '';
    if (t === 'slot' || t === 'guest' || src.uid){
      const uid = src.uid || src.slotUid || src.participantUid;
      const slots = window.slots || [];
      const slot = slots.find(s => s && s.uid === uid) || null;
      if (!slot) return { kind:'missing', label:`Missing slot ${uid||''}` };

      const vTrack = slot.videoTrack || slot.screenTrack || null;
      const label = slot.displayName || slot.name || uid || 'Guest';
      return { kind:'slot', uid, label, slot, vTrack };
    }

    if (t === 'return' || t === 'host' || src.isReturn === true){
      const vTrack = (typeof hostVideoTrack !== 'undefined') ? hostVideoTrack : null;
      if (!vTrack) return { kind:'missing', label:'RETURN' };
      return { kind:'return', label:'RETURN', vTrack };
    }

    if (t === 'media' || src.id){
      const id = src.id || src.mediaId;
      const items = window.mediaItems || [];
      const item = items.find(m => m && m.id === id) || null;
      if (!item) return { kind:'missing', label:`Missing media ${id||''}` };

      if (item.kind === 'video'){
        const v = getMediaVideoEl(item);
        return { kind:'media-video', id, label:item.name||'Video', item, vEl: v };
      }
      if (item.kind === 'image'){
        const img = getMediaImage(item);
        return { kind:'media-image', id, label:item.name||'Image', item, img };
      }
      if (item.kind === 'audio'){
        return { kind:'media-audio', id, label:item.name||'Audio', item };
      }
      if (item.kind === 'pdf'){
        return { kind:'media-pdf', id, label:item.name||'PDF', item };
      }
      return { kind:'media', id, label:item.name||'Media', item };
    }

    return { kind:'unknown', label:'Unknown source' };
  }

  function getLKVideoElForTrack(track){
    if (!track) return null;
    const key = track.sid || track.mediaStreamTrack?.id || track.name || ('t_' + Math.random().toString(16).slice(2));
    if (state.lkVideoEls.has(key)) return state.lkVideoEls.get(key);

    const wrap = ensureHiddenWrap();
    const v = document.createElement('video');
    v.playsInline = true;
    v.autoplay = true;
    v.muted = true;
    v.setAttribute('muted','');
    v.setAttribute('playsinline','');
    v.style.cssText = 'width:1px;height:1px;';

    try{
      track.attach(v);
    }catch(err){
      console.warn('[program_renderer] track.attach failed', err);
      return null;
    }

    wrap.appendChild(v);
    state.lkVideoEls.set(key, v);
    return v;
  }

  function getMediaVideoEl(item){
    if (!item || !item.id) return null;
    if (state.mediaVideoEls.has(item.id)) return state.mediaVideoEls.get(item.id);

    const wrap = ensureHiddenWrap();
    const v = document.createElement('video');
    v.playsInline = true;
    v.autoplay = true;
    v.loop = true;
    v.muted = true;
    v.setAttribute('muted','');
    v.setAttribute('playsinline','');
    v.crossOrigin = 'anonymous';
    v.src = item.url;
    v.style.cssText = 'width:1px;height:1px;';
    wrap.appendChild(v);

    // best-effort play
    const p = v.play();
    if (p && typeof p.catch === 'function') p.catch(()=>{});

    state.mediaVideoEls.set(item.id, v);
    return v;
  }

  function getMediaImage(item){
    if (!item || !item.id) return null;
    if (state.mediaImages.has(item.id)) return state.mediaImages.get(item.id);

    const img = new Image();
    img.crossOrigin = 'anonymous';
    img.src = item.url;
    state.mediaImages.set(item.id, img);
    return img;
  }

  function ensureBgImage(){
    const v = state.virtuals || {};
    if (v.mode !== 'image' || !v.imageDataUrl){
      state.bgImg = null;
      return;
    }
    if (state.bgImg && state.bgImg.src === v.imageDataUrl) return;
    const img = new Image();
    img.crossOrigin = 'anonymous';
    img.src = v.imageDataUrl;
    state.bgImg = img;
  }

  function drawCover(ctx, drawable, W, H){
    // drawable: <video> or <img>
    const sw = drawable.videoWidth || drawable.naturalWidth || 0;
    const sh = drawable.videoHeight || drawable.naturalHeight || 0;
    if (!sw || !sh) return false;

    const sAR = sw / sh;
    const dAR = W / H;

    let dw = W, dh = H, dx = 0, dy = 0;
    if (sAR > dAR){
      // wider
      dh = H;
      dw = H * sAR;
      dx = (W - dw) / 2;
      dy = 0;
    }else{
      dw = W;
      dh = W / sAR;
      dx = 0;
      dy = (H - dh) / 2;
    }
    ctx.drawImage(drawable, dx, dy, dw, dh);
    return true;
  }

  function drawPip(ctx, drawable, W, H){
    // PiP: large subject area with rounded mask + shadow
    const sw = drawable.videoWidth || drawable.naturalWidth || 0;
    const sh = drawable.videoHeight || drawable.naturalHeight || 0;
    if (!sw || !sh) return false;

    const margin = Math.round(W * 0.06);
    const pipW = W - margin * 2;
    const pipH = Math.round(pipW * 9 / 16);
    const x = margin;
    const y = H - margin - pipH;

    const sAR = sw / sh;
    const dAR = pipW / pipH;
    let dw = pipW, dh = pipH, dx = x, dy = y;

    if (sAR > dAR){
      dh = pipH;
      dw = pipH * sAR;
      dx = x + (pipW - dw) / 2;
      dy = y;
    }else{
      dw = pipW;
      dh = pipW / sAR;
      dx = x;
      dy = y + (pipH - dh) / 2;
    }

    const r = Math.round(Math.min(pipW, pipH) * 0.04);

    ctx.save();
    ctx.shadowColor = 'rgba(0,0,0,.55)';
    ctx.shadowBlur = Math.round(W * 0.025);
    ctx.shadowOffsetY = Math.round(H * 0.01);

    // rounded rect clip
    ctx.beginPath();
    const rr = (x0,y0,w,h,rad)=>{
      const r2 = Math.min(rad, w/2, h/2);
      ctx.moveTo(x0+r2, y0);
      ctx.arcTo(x0+w, y0, x0+w, y0+h, r2);
      ctx.arcTo(x0+w, y0+h, x0, y0+h, r2);
      ctx.arcTo(x0, y0+h, x0, y0, r2);
      ctx.arcTo(x0, y0, x0+w, y0, r2);
      ctx.closePath();
    };
    rr(x, y, pipW, pipH, r);
    ctx.clip();

    ctx.drawImage(drawable, dx, dy, dw, dh);
    ctx.restore();
    return true;
  }

  function drawBase(ctx, src, alpha){
    if (!src || src.kind === 'missing' || src.kind === 'unknown'){
      ctx.save();
      ctx.globalAlpha = alpha;
      ctx.fillStyle = '#000';
      ctx.fillRect(0,0,state.w,state.h);
      ctx.fillStyle = 'rgba(255,255,255,.75)';
      ctx.font = '600 32px system-ui, -apple-system, Segoe UI, Roboto, Arial';
      ctx.textAlign = 'center';
      ctx.fillText(src ? (src.label||'No source') : 'No source', state.w/2, state.h/2);
      ctx.restore();
      return;
    }

    const vset = state.virtuals || {};
    ensureBgImage();

    ctx.save();
    ctx.globalAlpha = alpha;

    if (vset.mode === 'image' && state.bgImg && state.bgImg.complete){
      // virtual set: background image + PiP video/image
      ctx.fillStyle = '#000';
      ctx.fillRect(0,0,state.w,state.h);
      drawCover(ctx, state.bgImg, state.w, state.h);

      const drawable = getDrawableForSource(src);
      if (drawable){
        drawPip(ctx, drawable, state.w, state.h);
      }else{
        // fallback label
        ctx.fillStyle = 'rgba(255,255,255,.8)';
        ctx.font = '600 28px system-ui, -apple-system, Segoe UI, Roboto, Arial';
        ctx.textAlign = 'center';
        ctx.fillText(src.label || 'Source', state.w/2, state.h/2);
      }
      ctx.restore();
      return;
    }

    // normal cover draw (optionally blurred)
    const drawable = getDrawableForSource(src);
    if (drawable){
      if (vset.mode === 'blur'){
        ctx.filter = `blur(${clamp(+vset.blur||12, 0, 40)}px)`;
        drawCover(ctx, drawable, state.w, state.h);
        ctx.filter = 'none';
        // overlay a slightly smaller sharp frame
        ctx.globalAlpha = alpha * 0.92;
        drawPip(ctx, drawable, state.w, state.h);
      }else{
        drawCover(ctx, drawable, state.w, state.h);
      }
    }else{
      ctx.fillStyle = '#000';
      ctx.fillRect(0,0,state.w,state.h);
      ctx.fillStyle = 'rgba(255,255,255,.8)';
      ctx.font = '600 28px system-ui, -apple-system, Segoe UI, Roboto, Arial';
      ctx.textAlign = 'center';
      ctx.fillText(src.label || 'Source', state.w/2, state.h/2);
    }

    ctx.restore();
  }

  function getDrawableForSource(src){
    if (!src) return null;
    if (src.kind === 'slot' || src.kind === 'return'){
      const t = src.vTrack;
      if (!t) return null;
      return getLKVideoElForTrack(t);
    }
    if (src.kind === 'media-video') return src.vEl || null;
    if (src.kind === 'media-image') return (src.img && src.img.complete) ? src.img : src.img;
    return null;
  }

  function drawOverlays(ctx, dtMs){
    const items = Array.isArray(state.overlays) ? state.overlays : [];
    if (!items.length) return;

    // Keep ticker moving
    const ticker = items.find(o => o && o.enabled && o.type === 'ticker') || null;
    if (ticker){
      const speed = clamp(+ticker.speed || 60, 5, 240); // px/sec
      state.tickerX -= (speed * (dtMs/1000));
      // Reset when off screen
      const text = String(ticker.text || '').trim();
      const tw = estimateTextWidth(ctx, text, 28) + state.w*0.35;
      if (state.tickerX < -tw) state.tickerX = state.w;
    }

    for (const o of items){
      if (!o || !o.enabled) continue;
      if (o.type === 'bug') drawBug(ctx, o);
      else if (o.type === 'lowerthird') drawLowerThird(ctx, o);
      else if (o.type === 'ticker') drawTicker(ctx, o);
    }
  }

  function estimateTextWidth(ctx, text, sizePx){
    ctx.save();
    ctx.font = `700 ${sizePx}px system-ui, -apple-system, Segoe UI, Roboto, Arial`;
    const w = ctx.measureText(text).width;
    ctx.restore();
    return w;
  }

  function accent(o){
    return (o && o.style && o.style.accent) ? o.style.accent : '#ff3b30';
  }

  function drawBug(ctx, o){
    const pad = 16;
    const w = Math.round(state.w * 0.18);
    const h = Math.round(state.h * 0.09);
    const x = state.w - w - pad;
    const y = pad;

    ctx.save();
    ctx.globalAlpha = 0.9;

    // plate
    ctx.fillStyle = 'rgba(0,0,0,.45)';
    roundRect(ctx, x, y, w, h, 12);
    ctx.fill();

    // accent bar
    ctx.fillStyle = accent(o);
    roundRect(ctx, x, y, Math.max(10, Math.round(w*0.06)), h, 12);
    ctx.fill();

    ctx.fillStyle = 'rgba(255,255,255,.95)';
    ctx.font = `800 ${Math.round(h*0.38)}px system-ui, -apple-system, Segoe UI, Roboto, Arial`;
    ctx.textAlign = 'left';
    ctx.textBaseline = 'middle';
    const txt = String(o.text||'LIVE').slice(0,18);
    ctx.fillText(txt, x + Math.round(w*0.10), y + h/2);

    ctx.restore();
  }

  function drawLowerThird(ctx, o){
    const pad = 18;
    const w = Math.round(state.w * 0.52);
    const h = Math.round(state.h * 0.16);
    const x = pad;
    const y = state.h - h - pad;

    ctx.save();
    ctx.globalAlpha = 0.92;

    // base
    ctx.fillStyle = 'rgba(0,0,0,.55)';
    roundRect(ctx, x, y, w, h, 16);
    ctx.fill();

    // accent strip
    ctx.fillStyle = accent(o);
    roundRect(ctx, x, y, Math.max(10, Math.round(w*0.02)), h, 16);
    ctx.fill();

    const title = String(o.text||'').trim().slice(0,64);
    const sub = String(o.subtext||'').trim().slice(0,80);

    ctx.fillStyle = 'rgba(255,255,255,.98)';
    ctx.textAlign = 'left';
    ctx.textBaseline = 'top';

    ctx.font = `800 ${Math.round(h*0.34)}px system-ui, -apple-system, Segoe UI, Roboto, Arial`;
    ctx.fillText(title || '—', x + Math.round(w*0.06), y + Math.round(h*0.18));

    ctx.fillStyle = 'rgba(255,255,255,.86)';
    ctx.font = `650 ${Math.round(h*0.22)}px system-ui, -apple-system, Segoe UI, Roboto, Arial`;
    ctx.fillText(sub, x + Math.round(w*0.06), y + Math.round(h*0.58));

    ctx.restore();
  }

  function drawTicker(ctx, o){
    const h = Math.round(state.h * 0.075);
    const y = state.h - h;
    const text = String(o.text || '').trim();
    if (!text) return;

    ctx.save();
    ctx.globalAlpha = 0.92;

    // bar
    ctx.fillStyle = 'rgba(0,0,0,.55)';
    ctx.fillRect(0, y, state.w, h);

    // accent top line
    ctx.fillStyle = accent(o);
    ctx.fillRect(0, y, state.w, Math.max(3, Math.round(h*0.07)));

    // text
    ctx.fillStyle = 'rgba(255,255,255,.95)';
    ctx.font = `800 ${Math.round(h*0.52)}px system-ui, -apple-system, Segoe UI, Roboto, Arial`;
    ctx.textAlign = 'left';
    ctx.textBaseline = 'middle';

    const x = state.tickerX;
    ctx.fillText(text, x, y + h/2);

    // repeat to create continuous flow
    const tw = ctx.measureText(text).width + state.w*0.35;
    ctx.fillText(text, x + tw, y + h/2);

    ctx.restore();
  }

  function roundRect(ctx, x, y, w, h, r){
    const rr = Math.min(r, w/2, h/2);
    ctx.beginPath();
    ctx.moveTo(x+rr, y);
    ctx.arcTo(x+w, y, x+w, y+h, rr);
    ctx.arcTo(x+w, y+h, x, y+h, rr);
    ctx.arcTo(x, y+h, x, y, rr);
    ctx.arcTo(x, y, x+w, y, rr);
    ctx.closePath();
  }

  function startTransition(toResolved){
    const kind = (state.transition && state.transition.kind) ? state.transition.kind : 'cut';
    const dur = clamp(+((state.transition && state.transition.fadeMs) || 250), 0, 4000);

    state.transFrom = state.programSource;
    state.transTo = toResolved;

    state.transKind = kind;
    state.transStart = now();
    state.transDur = (kind === 'cut') ? 0 : dur;

    // immediate cut
    if (kind === 'cut' || dur === 0){
      state.programSource = toResolved;
      state.pendingSource = null;
      state.transFrom = null;
      state.transTo = null;
      state.transStart = 0;
      state.transDur = 0;
    }
  }

  function takeProgram(){
    if (!ensureCanvas()) return;
    const src = resolveSource(window.previewSource);
    if (!src) return;

    startTransition(src);
    CP.bus.emit('program:changed', { at: now(), source: src });
  }

  function setTransition(payload){
    if (!payload || typeof payload !== 'object') return;
    state.transition = Object.assign({}, state.transition, payload);
  }

  function setOverlays(payload){
    if (!payload) return;
    state.overlays = Array.isArray(payload.overlays) ? payload.overlays : (Array.isArray(payload) ? payload : []);
  }

  function setVirtuals(payload){
    if (!payload) return;
    const v = payload.virtuals ? payload.virtuals : payload;
    if (v && typeof v === 'object'){
      state.virtuals = Object.assign({}, state.virtuals, v);
      ensureBgImage();
    }
  }

  function drawLoop(ts){
    if (!state.enabled || !state.ctx) { requestAnimationFrame(drawLoop); return; }

    const ctx = state.ctx;
    const W = state.w, H = state.h;

    const last = state.lastTs || ts;
    const dtMs = clamp(ts - last, 0, 100);
    state.lastTs = ts;

    // clear
    ctx.setTransform(1,0,0,1,0,0);
    ctx.clearRect(0,0,W,H);

    // If never taken, show instruction
    if (!state.programSource && !state.transTo){
      ctx.fillStyle = '#000';
      ctx.fillRect(0,0,W,H);
      ctx.fillStyle = 'rgba(255,255,255,.78)';
      ctx.font = '700 30px system-ui, -apple-system, Segoe UI, Roboto, Arial';
      ctx.textAlign = 'center';
      ctx.fillText('PROGRAM EMPTY — click TAKE', W/2, H/2);
      // overlays still allowed
      drawOverlays(ctx, dtMs);
      requestAnimationFrame(drawLoop);
      return;
    }

    // Transition handling
    if (state.transTo && state.transKind !== 'cut' && state.transDur > 0){
      const t = clamp((now() - state.transStart) / state.transDur, 0, 1);
      const from = state.transFrom || state.programSource;
      const to = state.transTo;

      if (state.transKind === 'dissolve'){
        drawBase(ctx, from, 1 - t);
        drawBase(ctx, to, t);
      }else if (state.transKind === 'dip'){
        // fade to black then from black
        if (t < 0.5){
          const a = 1 - (t / 0.5);
          drawBase(ctx, from, a);
          ctx.fillStyle = `rgba(0,0,0,${t/0.5})`;
          ctx.fillRect(0,0,W,H);
        }else{
          const tt = (t - 0.5) / 0.5;
          ctx.fillStyle = `rgba(0,0,0,${1-tt})`;
          ctx.fillRect(0,0,W,H);
          drawBase(ctx, to, tt);
        }
      }else if (state.transKind === 'wipe-l' || state.transKind === 'wipe-r'){
        drawBase(ctx, from, 1);
        ctx.save();
        const x = (state.transKind === 'wipe-l') ? (W * t) : (W * (1 - t));
        if (state.transKind === 'wipe-l'){
          ctx.beginPath(); ctx.rect(0,0,x,H); ctx.clip();
        }else{
          ctx.beginPath(); ctx.rect(x,0,W-x,H); ctx.clip();
        }
        drawBase(ctx, to, 1);
        ctx.restore();
      }else{
        // fallback dissolve
        drawBase(ctx, from, 1 - t);
        drawBase(ctx, to, t);
      }

      if (t >= 1){
        state.programSource = state.transTo;
        state.transFrom = null;
        state.transTo = null;
        state.transStart = 0;
        state.transDur = 0;
      }
    }else{
      // steady state
      drawBase(ctx, state.programSource, 1);
    }

    // Overlays after base/transition
    drawOverlays(ctx, dtMs);

    requestAnimationFrame(drawLoop);
  }

  // Bus wiring — consumes UI plugin events
  CP.bus.on('program:take', takeProgram);
  CP.bus.on('transition:run', setTransition);
  CP.bus.on('overlays:apply', setOverlays);
  CP.bus.on('virtuals:apply', setVirtuals);

  // Optional: let core broadcast preview changes (if we can wrap it safely)
  if (typeof window.setPreviewSource === 'function' && !window.__cp_wrapped_setPreviewSource){
    window.__cp_wrapped_setPreviewSource = true;
    const orig = window.setPreviewSource;
    window.setPreviewSource = function(src){
      try{
        orig.apply(this, arguments);
      }finally{
        try{ CP.bus.emit('preview:changed', { at: now(), source: window.previewSource }); }catch(_){}
      }
    };
  }

  // Init now (best effort)
  ensureCanvas();

  // Public API
  CP.program = {
    take: takeProgram,
    setTransition,
    setOverlays,
    setVirtuals,
    getState: () => ({
      enabled: state.enabled,
      transition: state.transition,
      overlays: state.overlays,
      virtuals: state.virtuals,
      program: state.programSource ? { kind: state.programSource.kind, label: state.programSource.label } : null
    }),
    _internals: state
  };

  console.log('[program_renderer] loaded');
})();