/* =========================================================
   Co-Producer Plugin Host — v1
   File: addons/plugin_host.js

   Goal:
   - One-time include in coproducer_livekit.html
   - After that, new features ship as independent plugins in /addons
   - Plugins are loaded from addons/plugins.json (manifest)

   Provides:
   - window.CP.version
   - CP.bus (event emitter)
   - CP.store (localStorage helpers)
   - CP.dom.createModal(), CP.dom.addDockButton()
   - CP.hookClick(selector, handler, mode)

========================================================= */
(function(){
  "use strict";

  const qs = (k, def="") => {
    try{ return (new URL(location.href)).searchParams.get(k) ?? def; }catch(_){ return def; }
  };
  const log = (...a)=>console.log("[CP:host]", ...a);
  const warn = (...a)=>console.warn("[CP:host]", ...a);
  const err  = (...a)=>console.error("[CP:host]", ...a);

  function createBus(){
    const map = new Map();
    return {
      on(ev, fn){
        if (!map.has(ev)) map.set(ev, new Set());
        map.get(ev).add(fn);
        return ()=>{ try{ map.get(ev)?.delete(fn); }catch(_){ } };
      },
      emit(ev, data){
        const set = map.get(ev);
        if (!set) return;
        for (const fn of Array.from(set)){
          try{ fn(data); }catch(e){ err("bus handler error", ev, e); }
        }
      }
    };
  }

  const store = {
    get(k, def=null){
      try{ const v = localStorage.getItem(k); return (v==null) ? def : v; }catch(_){ return def; }
    },
    set(k, v){ try{ localStorage.setItem(k, String(v)); }catch(_){ } },
    getJSON(k, def=null){
      try{ const v = localStorage.getItem(k); return v ? JSON.parse(v) : def; }catch(_){ return def; }
    },
    setJSON(k, obj){ try{ localStorage.setItem(k, JSON.stringify(obj)); }catch(_){ } },
    del(k){ try{ localStorage.removeItem(k); }catch(_){ } }
  };

  function ensureHostStyles(){
    if (document.getElementById("cpHostStyles")) return;
    const css = `
      .cpHostModalBackdrop{position:fixed;inset:0;display:none;align-items:center;justify-content:center;background:rgba(0,0,0,.72);z-index:99999}
      .cpHostModalBackdrop.show{display:flex}
      .cpHostModal{width:min(980px,calc(100% - 28px));border-radius:18px;border:1px solid rgba(255,255,255,.14);background:linear-gradient(180deg,rgba(27,36,64,.96),rgba(12,16,32,.96));box-shadow:0 26px 70px rgba(0,0,0,.65);overflow:hidden}
      .cpHostModalTop{display:flex;align-items:center;justify-content:space-between;gap:12px;padding:12px;border-bottom:1px solid rgba(255,255,255,.10);background:rgba(0,0,0,.18)}
      .cpHostModalTop .title{font-weight:1000;letter-spacing:.2px;color:rgba(255,255,255,.95)}
      .cpHostModalTop .right{display:flex;align-items:center;gap:10px}
      .cpHostPill{display:inline-flex;align-items:center;justify-content:center;padding:5px 10px;border-radius:999px;font-size:12px;font-weight:900;border:1px solid rgba(255,255,255,.14);color:rgba(255,255,255,.90);background:rgba(0,0,0,.22)}
      .cpHostBtn{border:1px solid rgba(255,255,255,.14);background:rgba(255,255,255,.08);color:rgba(255,255,255,.92);padding:9px 12px;border-radius:12px;cursor:pointer;font-weight:900}
      .cpHostBtn:hover{filter:brightness(1.08)}
      .cpHostModalBody{padding:14px}
      .cpHostMuted{color:rgba(255,255,255,.66);font-size:12px}
      .cpPluginDock{position:fixed;top:92px;right:18px;display:flex;flex-direction:column;gap:10px;z-index:90000}
      .cpDockBtn{width:56px;height:56px;border-radius:999px;border:1px solid rgba(255,255,255,.16);background:rgba(255,255,255,.08);box-shadow:0 14px 34px rgba(0,0,0,.55);display:flex;align-items:center;justify-content:center;cursor:pointer;user-select:none;font-size:20px}
      .cpDockBtn:hover{background:rgba(255,255,255,.12)}
    `;
    const style = document.createElement("style");
    style.id = "cpHostStyles";
    style.textContent = css;
    document.head.appendChild(style);
  }

  function createModal({ id, title, pillText }){
    ensureHostStyles();
    const existing = document.getElementById(id);
    if (existing) return existing._cpModalApi;

    const backdrop = document.createElement("div");
    backdrop.className = "cpHostModalBackdrop";
    backdrop.id = id;

    const modal = document.createElement("div");
    modal.className = "cpHostModal";

    const top = document.createElement("div");
    top.className = "cpHostModalTop";
    top.innerHTML = `
      <div class="title">${title || "Plugin"}</div>
      <div class="right">
        <span class="cpHostPill" id="${id}__pill">${pillText || "READY"}</span>
        <button class="cpHostBtn" id="${id}__close" type="button">Close</button>
      </div>
    `;

    const body = document.createElement("div");
    body.className = "cpHostModalBody";

    modal.appendChild(top);
    modal.appendChild(body);
    backdrop.appendChild(modal);
    document.body.appendChild(backdrop);

    const pillEl = document.getElementById(id + "__pill");
    const closeEl = document.getElementById(id + "__close");

    function open(){ backdrop.classList.add("show"); backdrop.setAttribute("aria-hidden","false"); }
    function close(){ backdrop.classList.remove("show"); backdrop.setAttribute("aria-hidden","true"); }
    function setPill(text){ if (pillEl) pillEl.textContent = String(text || ""); }

    backdrop.addEventListener("click", (e)=>{ if (e.target === backdrop) close(); });
    if (closeEl) closeEl.addEventListener("click", close);

    const api = { backdrop, modal, top, body, open, close, setPill };
    backdrop._cpModalApi = api;
    return api;
  }

  function ensureDock(){
    ensureHostStyles();
    let dock = document.querySelector(".cpPluginDock");
    if (dock) return dock;
    dock = document.createElement("div");
    dock.className = "cpPluginDock";
    document.body.appendChild(dock);
    return dock;
  }

  function addDockButton({ id, icon="★", title="Plugin", onClick }){
    const dock = ensureDock();
    const btn = document.createElement("button");
    btn.className = "cpDockBtn";
    btn.id = id;
    btn.type = "button";
    btn.title = title;
    btn.setAttribute("aria-label", title);
    btn.textContent = icon;
    btn.addEventListener("click", ()=>{ try{ onClick && onClick(); }catch(e){ err(e); } });
    dock.appendChild(btn);
    return btn;
  }

  function hookClick(selectorOrEl, handler, mode="replace"){
    const el = (typeof selectorOrEl === "string") ? document.querySelector(selectorOrEl) : selectorOrEl;
    if (!el) return { ok:false, reason:"not_found" };
    const prev = el.onclick;
    el.onclick = null;
    el.addEventListener("click", (e)=>{
      if (mode === "append"){
        try{ handler && handler(e); }catch(ex){ err(ex); }
        try{ prev && prev.call(el, e); }catch(ex){ err(ex); }
      } else {
        e.preventDefault(); e.stopPropagation();
        try{ handler && handler(e); }catch(ex){ err(ex); }
      }
    }, true);
    return { ok:true };
  }

  function loadScript(src){
    return new Promise((resolve, reject)=>{
      const s = document.createElement("script");
      s.src = src;
      s.async = false;
      s.onload = ()=>resolve(true);
      s.onerror = ()=>reject(new Error("Failed to load: " + src));
      document.head.appendChild(s);
    });
  }

  async function fetchManifest(url){
    const res = await fetch(url, { credentials:"same-origin" });
    if (!res.ok) throw new Error("Manifest fetch failed: " + res.status);
    return await res.json();
  }

  function parsePluginFilter(){
    const only = (qs("plugins","") || "").split(",").map(s=>s.trim()).filter(Boolean);
    const dis  = (qs("disablePlugins","") || "").split(",").map(s=>s.trim()).filter(Boolean);
    return { only, dis };
  }

  const registry = new Map();
  const bus = createBus();

  function registerPlugin(plugin){
    if (!plugin || !plugin.id) throw new Error("Plugin missing id");
    if (registry.has(plugin.id)) { warn("plugin already registered:", plugin.id); return; }
    registry.set(plugin.id, plugin);
    try{
      plugin.init && plugin.init(CP);
      bus.emit("plugin:registered", { id: plugin.id });
      log("registered:", plugin.id);
    }catch(e){
      err("plugin init failed:", plugin.id, e);
    }
  }

  const CP = {
    version: "1.0.0",
    bus,
    store,
    dom: { createModal, addDockButton, ensureDock },
    hookClick,
    registerPlugin
  };

  window.CP = CP;

  async function boot(){
    const manifestUrl = window.CP_PLUGIN_MANIFEST_URL || "addons/plugins.json";
    const { only, dis } = parsePluginFilter();

    let manifest;
    try{ manifest = await fetchManifest(manifestUrl); }
    catch(e){
      warn("No manifest (or invalid). Plugins not loaded.", e);
      bus.emit("plugins:manifest_error", { error: String(e) });
      return;
    }

    const plugins = Array.isArray(manifest.plugins) ? manifest.plugins : [];
    const toLoad = [];

    for (const p of plugins){
      if (!p || !p.id) continue;
      if (p.enabled === false) continue;
      if (only.length && !only.includes(p.id)) continue;
      if (dis.length && dis.includes(p.id)) continue;
      const scripts = Array.isArray(p.scripts) ? p.scripts : [];
      if (!scripts.length) continue;
      toLoad.push({ id:p.id, scripts });
    }

    bus.emit("plugins:loading", { count: toLoad.length });

    for (const item of toLoad){
      for (const src of item.scripts){
        try{ await loadScript(src); }
        catch(e){
          err("plugin script failed", item.id, src, e);
          bus.emit("plugin:load_error", { id:item.id, src, error:String(e) });
          break;
        }
      }
    }

    bus.emit("plugins:loaded", { count: toLoad.length });
  }

  if (document.readyState === "complete" || document.readyState === "interactive") boot();
  else document.addEventListener("DOMContentLoaded", boot, { once:true });
})();
