/* =========================================================
   Overlays Plugin — UI (v1)
   File: addons/overlays.ui.js
========================================================= */
(function(){
  "use strict";

  function esc(s){
    return String(s).replace(/[&<>"']/g, (c)=>({ "&":"&amp;","<":"&lt;",">":"&gt;",'"':"&quot;","'":"&#39;" }[c]));
  }

  function ensureStyles(){
    if (document.getElementById("cpOverlaysStyles")) return;
    const css = `
      .ovWrap{ padding:14px; }
      .ovTop{ display:flex; align-items:center; justify-content:space-between; gap:12px; flex-wrap:wrap; margin-bottom:12px; }
      .ovPill{ display:inline-flex; align-items:center; gap:8px; padding:6px 10px; border-radius:999px;
        border:1px solid rgba(255,255,255,.12); background:rgba(0,0,0,.18); font-size:12px; font-weight:900; }
      .ovPill b{ font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono","Courier New", monospace; }
      .ovBtns{ display:flex; gap:10px; flex-wrap:wrap; justify-content:flex-end; }
      .ovBtn{ border:1px solid rgba(255,255,255,.14); background:rgba(255,255,255,.08); color:rgba(255,255,255,.92);
        padding:10px 12px; border-radius:12px; cursor:pointer; font-weight:1000; }
      .ovBtn:hover{ filter:brightness(1.08); }
      .ovBtn.primary{ border-color: rgba(74,163,255,.45); background: rgba(74,163,255,.14); }
      .ovBtn.ok{ border-color: rgba(32,201,151,.45); background: rgba(32,201,151,.12); }
      .ovBtn.danger{ border-color: rgba(255,74,106,.45); background: rgba(255,74,106,.12); }

      .ovGrid{ display:grid; grid-template-columns: 1fr; gap:12px; }
      .ovList{ border:1px solid rgba(255,255,255,.10); border-radius:14px; overflow:hidden; background:rgba(0,0,0,.12); }
      .ovRow{ display:grid; grid-template-columns: 28px 1fr 110px 70px 46px; gap:10px; align-items:center;
        padding:10px 12px; border-bottom:1px solid rgba(255,255,255,.08); }
      .ovRow:last-child{ border-bottom:none; }
      .ovChk{ width:18px; height:18px; accent-color: rgba(74,163,255,.95); }
      .ovName{ font-weight:1000; }
      .ovMeta{ color:rgba(255,255,255,.62); font-size:11px; white-space:nowrap; overflow:hidden; text-overflow:ellipsis; }
      .ovSel, .ovInp{ width:100%; padding:8px 10px; border-radius:12px; border:1px solid rgba(255,255,255,.12);
        background:rgba(0,0,0,.22); color:rgba(255,255,255,.92); outline:none; }
      .ovSel:focus, .ovInp:focus{ border-color: rgba(74,163,255,.55); box-shadow:0 0 0 4px rgba(74,163,255,.12); }
      .ovMini{ height:34px; width:46px; border-radius:12px; border:1px solid rgba(255,255,255,.14);
        background:linear-gradient(180deg, rgba(255,255,255,.10), rgba(0,0,0,.22)); color:rgba(255,255,255,.92);
        font-weight:1000; cursor:pointer; }
      .ovMini:hover{ filter:brightness(1.08); }

      .ovEdit{ border:1px solid rgba(255,255,255,.10); border-radius:14px; background:rgba(0,0,0,.12); padding:12px; }
      .ovEditHdr{ display:flex; align-items:center; justify-content:space-between; gap:10px; flex-wrap:wrap; margin-bottom:10px; }
      .ovEditHdr b{ font-size:12px; letter-spacing:.10em; text-transform:uppercase; }
      .ovHelp{ color:rgba(255,255,255,.62); font-size:11px; }
      .ovForm{ display:grid; grid-template-columns: 120px 1fr; gap:10px 12px; align-items:center; }
      .ovLbl{ color:rgba(255,255,255,.74); font-size:12px; }
      .ovPre{ margin-top:10px; padding:10px; border-radius:12px; border:1px solid rgba(255,255,255,.10);
        background:rgba(0,0,0,.22); font-size:12px; max-height:220px; overflow:auto; }
      .ovNote{ color:rgba(255,255,255,.66); font-size:12px; line-height:1.35; margin-top:10px; }
    `;
    const st = document.createElement("style");
    st.id = "cpOverlaysStyles";
    st.textContent = css;
    document.head.appendChild(st);
  }

  function findRendererFrames(){
    const frames = [];
    const byId = ["programRenderer","programFrame","rendererFrame","previewFrame","programPreview"];
    for (const id of byId){
      const el = document.getElementById(id);
      if (el && el.tagName === "IFRAME") frames.push(el);
    }
    document.querySelectorAll("iframe[data-program-renderer], iframe[data-renderer], iframe").forEach(f=>{
      if (f && !frames.includes(f)) frames.push(f);
    });
    return frames;
  }

  function postToRenderer(type, payload){
    const frames = findRendererFrames();
    for (const f of frames){
      try{ f.contentWindow && f.contentWindow.postMessage({ type, payload }, "*"); }catch(_){}
    }
    return frames.length;
  }

  function init(CP){
    if (!window.CPOverlays){
      console.warn("[overlays] CPOverlays missing (logic not loaded?)");
      return;
    }
    ensureStyles();

    const api = window.CPOverlays;
    const modal = CP.dom.createModal({ id:"cp_overlays_modal", title:"Overlays", pillText:"READY" });

    let editingId = "";

    function build(){
      api.load(); // ensure defaults
      modal.body.innerHTML = `
        <div class="ovWrap">
          <div class="ovTop">
            <div class="ovPill">Active payload: <b id="ovCount">0</b></div>
            <div class="ovBtns">
              <button class="ovBtn primary" id="ovApplyBtn" type="button">APPLY TO PREVIEW</button>
              <button class="ovBtn ok" id="ovCopyBtn" type="button">COPY JSON</button>
              <button class="ovBtn" id="ovAddBtn" type="button">ADD</button>
              <button class="ovBtn danger" id="ovResetBtn" type="button">RESET</button>
            </div>
          </div>

          <div class="ovGrid">
            <div class="ovList" id="ovList"></div>

            <div class="ovEdit">
              <div class="ovEditHdr">
                <b>Edit</b>
                <span class="ovHelp" id="ovEditHint">Select ✎ to edit an overlay.</span>
              </div>
              <div class="ovForm">
                <div class="ovLbl">Name</div>
                <input class="ovInp" id="ovName" placeholder="Lower Third" />
                <div class="ovLbl">Type</div>
                <select class="ovSel" id="ovType">
                  <option value="lowerthird">lowerthird</option>
                  <option value="ticker">ticker</option>
                  <option value="bug">bug</option>
                  <option value="custom">custom</option>
                </select>
                <div class="ovLbl">Position</div>
                <select class="ovSel" id="ovPos">
                  <option value="bl">bottom-left</option>
                  <option value="br">bottom-right</option>
                  <option value="b">bottom</option>
                  <option value="tl">top-left</option>
                  <option value="tr">top-right</option>
                  <option value="t">top</option>
                </select>
                <div class="ovLbl">Text</div>
                <input class="ovInp" id="ovText" placeholder="Main text" />
                <div class="ovLbl">Subtext</div>
                <input class="ovInp" id="ovSub" placeholder="Optional" />
                <div class="ovLbl">Accent</div>
                <input class="ovInp" id="ovAccent" placeholder="#4aa3ff" />
                <div class="ovLbl">Speed</div>
                <input class="ovInp" id="ovSpeed" placeholder="1.0 (ticker)" />
              </div>

              <div class="ovBtns" style="margin-top:10px;">
                <button class="ovBtn ok" id="ovSaveBtn" type="button">SAVE</button>
                <button class="ovBtn danger" id="ovDelBtn" type="button" disabled>DELETE</button>
              </div>

              <pre class="ovPre" id="ovPayload"></pre>
              <div class="ovNote">
                Preview integration: sends <b>postMessage</b> to any detected iframe.
                Add <span style="font-family:ui-monospace">addons/renderer_bridge.js</span> in your program renderer page to receive commands.
              </div>
            </div>
          </div>
        </div>
      `;

      const listEl = document.getElementById("ovList");

      function refreshList(){
        const cur = api.load();
        listEl.innerHTML = "";
        for (const o of cur){
          const row = document.createElement("div");
          row.className = "ovRow";
          row.innerHTML = `
            <input class="ovChk" type="checkbox" data-id="${esc(o.id)}" ${o.enabled ? "checked":""} />
            <div>
              <div class="ovName">${esc(o.name || o.type)}</div>
              <div class="ovMeta">${esc(o.type)} • ${esc(o.pos || "bl")} • ${esc((o.text||"").slice(0,60))}</div>
            </div>
            <select class="ovSel" data-pos="${esc(o.id)}">
              ${["bl","br","b","tl","tr","t"].map(p=>`<option value="${p}" ${p===o.pos?"selected":""}>${p}</option>`).join("")}
            </select>
            <input class="ovInp" data-text="${esc(o.id)}" value="${esc(o.text||"")}" placeholder="Text" />
            <button class="ovMini" data-edit="${esc(o.id)}" title="Edit" type="button">✎</button>
          `;
          listEl.appendChild(row);
        }

        listEl.querySelectorAll("input[data-id]").forEach(chk=>{
          chk.addEventListener("change", ()=>{
            const id = chk.getAttribute("data-id");
            const arr = api.load();
            const it = arr.find(x=>x.id===id);
            if (it){ it.enabled = !!chk.checked; api.save(arr); updatePayload(); }
          });
        });
        listEl.querySelectorAll("select[data-pos]").forEach(sel=>{
          sel.addEventListener("change", ()=>{
            const id = sel.getAttribute("data-pos");
            const arr = api.load();
            const it = arr.find(x=>x.id===id);
            if (it){ it.pos = sel.value; api.save(arr); updatePayload(); }
          });
        });
        listEl.querySelectorAll("input[data-text]").forEach(inp=>{
          inp.addEventListener("input", ()=>{
            const id = inp.getAttribute("data-text");
            const arr = api.load();
            const it = arr.find(x=>x.id===id);
            if (it){ it.text = inp.value; api.save(arr); updatePayload(); }
          });
        });
        listEl.querySelectorAll("button[data-edit]").forEach(btn=>{
          btn.addEventListener("click", ()=>edit(btn.getAttribute("data-edit")||""));
        });
      }

      function edit(id){
        const cur = api.load();
        const it = cur.find(x=>x.id===id);
        if (!it) return;
        editingId = it.id;
        document.getElementById("ovEditHint").textContent = "Editing: " + (it.name || it.id);
        document.getElementById("ovName").value = it.name || "";
        document.getElementById("ovType").value = it.type || "custom";
        document.getElementById("ovPos").value = it.pos || "bl";
        document.getElementById("ovText").value = it.text || "";
        document.getElementById("ovSub").value = it.subtext || "";
        document.getElementById("ovAccent").value = (it.style && it.style.accent) ? it.style.accent : "";
        document.getElementById("ovSpeed").value = (typeof it.speed === "number") ? String(it.speed) : "";
        document.getElementById("ovDelBtn").disabled = false;
      }

      function clearEdit(){
        editingId = "";
        document.getElementById("ovEditHint").textContent = "Select ✎ to edit an overlay.";
        document.getElementById("ovName").value = "";
        document.getElementById("ovType").value = "lowerthird";
        document.getElementById("ovPos").value = "bl";
        document.getElementById("ovText").value = "";
        document.getElementById("ovSub").value = "";
        document.getElementById("ovAccent").value = "";
        document.getElementById("ovSpeed").value = "";
        document.getElementById("ovDelBtn").disabled = true;
      }

      function updatePayload(){
        const p = api.payload();
        document.getElementById("ovPayload").textContent = JSON.stringify(p, null, 2);
        document.getElementById("ovCount").textContent = String((p.overlays||[]).length);
      }

      function saveEdit(){
        if (!editingId){ alert("Pick an overlay to edit, or click ADD."); return; }
        const list = api.load();
        const it = list.find(x=>x.id===editingId);
        if (!it) return;
        it.name = document.getElementById("ovName").value.trim() || it.name;
        it.type = document.getElementById("ovType").value;
        it.pos = document.getElementById("ovPos").value;
        it.text = document.getElementById("ovText").value || "";
        it.subtext = document.getElementById("ovSub").value || "";
        const a = (document.getElementById("ovAccent").value||"").trim();
        it.style = it.style && typeof it.style==="object" ? it.style : {};
        if (a) it.style.accent = a;
        const sp = parseFloat(document.getElementById("ovSpeed").value);
        if (Number.isFinite(sp)) it.speed = sp;
        api.save(list);
        refreshList();
        updatePayload();
      }

      function delEdit(){
        if (!editingId) return;
        if (!confirm("Delete this overlay?")) return;
        let list = api.load();
        list = list.filter(x=>x.id!==editingId);
        api.save(list);
        clearEdit();
        refreshList();
        updatePayload();
      }

      function addNew(){
        const list = api.load();
        const id = api.uid();
        list.push({ id, type:"lowerthird", name:"New Overlay", enabled:true, text:"", subtext:"", pos:"bl", speed:1.0, style:{ accent:"#4aa3ff" } });
        api.save(list);
        refreshList();
        updatePayload();
        edit(id);
      }

      async function copyJSON(){
        const txt = document.getElementById("ovPayload").textContent || "";
        try{ await navigator.clipboard.writeText(txt); }catch(_){}
      }

      function apply(){
        const p = api.payload();
        const count = postToRenderer("cp:overlays:set", p);
        CP.bus.emit("overlays:apply", { payload: p, frames: count });
        modal.setPill(count ? "SENT" : "NO RENDERER");
      }

      function reset(){
        if (!confirm("Reset overlays to defaults?")) return;
        api.save(null);
        api.load();
        clearEdit();
        refreshList();
        updatePayload();
      }

      document.getElementById("ovApplyBtn").addEventListener("click", apply);
      document.getElementById("ovCopyBtn").addEventListener("click", copyJSON);
      document.getElementById("ovAddBtn").addEventListener("click", addNew);
      document.getElementById("ovResetBtn").addEventListener("click", reset);
      document.getElementById("ovSaveBtn").addEventListener("click", saveEdit);
      document.getElementById("ovDelBtn").addEventListener("click", delEdit);

      refreshList();
      updatePayload();
      clearEdit();
    }

    function open(){ build(); modal.open(); }

    const hook = CP.hookClick("#qaOverlays", open, "replace");
    if (!hook.ok) CP.dom.addDockButton({ id:"cpDockOverlays", icon:"🧩", title:"Overlays", onClick: open });

    window.CPOverlaysUI = { open };
  }

  if (window.CP && typeof window.CP.registerPlugin === "function"){
    window.CP.registerPlugin({ id:"overlays", init });
  } else {
    console.warn("[overlays] CP plugin host not found. Include addons/plugin_host.js first.");
  }
})();
