/* =========================================================
   Transitions Plugin — UI (v1)
   File: addons/transitions.ui.js
========================================================= */
(function(){
  "use strict";

  function ensureStyles(){
    if (document.getElementById("cpTransitionsStyles")) return;
    const css = `
      .trWrap{ padding:14px; }
      .trTop{ display:flex; align-items:center; justify-content:space-between; gap:12px; flex-wrap:wrap; margin-bottom:12px; }
      .trPill{ display:inline-flex; align-items:center; gap:8px; padding:6px 10px; border-radius:999px;
        border:1px solid rgba(255,255,255,.12); background:rgba(0,0,0,.18); font-size:12px; font-weight:900; }
      .trPill b{ font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono","Courier New", monospace; }
      .trTake{
        width: 100%;
        padding: 16px 14px;
        border-radius: 16px;
        border: 1px solid rgba(255,255,255,.18);
        cursor: pointer;
        font-weight: 1100;
        letter-spacing: .12em;
        text-transform: uppercase;
        background: linear-gradient(180deg, rgba(255,74,106,.16), rgba(0,0,0,.22));
        box-shadow: 0 18px 44px rgba(0,0,0,.55), inset 0 1px 0 rgba(255,255,255,.08);
      }
      .trTake:hover{ filter: brightness(1.08); }
      .trGrid{ display:grid; grid-template-columns: 1fr; gap:12px; margin-top:12px; }
      .trCard{ border:1px solid rgba(255,255,255,.10); border-radius:14px; background:rgba(0,0,0,.12); padding:12px; }
      .trRow{ display:grid; grid-template-columns: 140px 1fr; gap:10px 12px; align-items:center; }
      .trLbl{ color:rgba(255,255,255,.74); font-size:12px; }
      .trSel, .trInp{ width:100%; padding:10px 10px; border-radius:12px; border:1px solid rgba(255,255,255,.12);
        background:rgba(0,0,0,.22); color:rgba(255,255,255,.92); outline:none; }
      .trSel:focus, .trInp:focus{ border-color: rgba(74,163,255,.55); box-shadow:0 0 0 4px rgba(74,163,255,.12); }
      .trBtns{ display:flex; gap:10px; justify-content:flex-end; flex-wrap:wrap; margin-top:10px; }
      .trBtn{ border:1px solid rgba(255,255,255,.14); background:rgba(255,255,255,.08); color:rgba(255,255,255,.92);
        padding:10px 12px; border-radius:12px; cursor:pointer; font-weight:1000; }
      .trBtn.primary{ border-color: rgba(74,163,255,.45); background: rgba(74,163,255,.14); }
      .trBtn:hover{ filter:brightness(1.08); }
      .trPre{ margin-top:10px; padding:10px; border-radius:12px; border:1px solid rgba(255,255,255,.10);
        background:rgba(0,0,0,.22); font-size:12px; max-height:220px; overflow:auto; }
      .trNote{ color:rgba(255,255,255,.66); font-size:12px; line-height:1.35; margin-top:10px; }
    `;
    const st = document.createElement("style");
    st.id = "cpTransitionsStyles";
    st.textContent = css;
    document.head.appendChild(st);
  }

  function findRendererFrames(){
    const frames = [];
    const byId = ["programRenderer","programFrame","rendererFrame","previewFrame","programPreview"];
    for (const id of byId){
      const el = document.getElementById(id);
      if (el && el.tagName === "IFRAME") frames.push(el);
    }
    document.querySelectorAll("iframe[data-program-renderer], iframe[data-renderer], iframe").forEach(f=>{
      if (f && !frames.includes(f)) frames.push(f);
    });
    return frames;
  }

  function postToRenderer(type, payload){
    const frames = findRendererFrames();
    for (const f of frames){
      try{ f.contentWindow && f.contentWindow.postMessage({ type, payload }, "*"); }catch(_){}
    }
    return frames.length;
  }

  function init(CP){
    if (!window.CPTransitions){
      console.warn("[transitions] CPTransitions missing (logic not loaded?)");
      return;
    }
    ensureStyles();

    const api = window.CPTransitions;
    const modal = CP.dom.createModal({ id:"cp_transitions_modal", title:"Transitions", pillText:"READY" });

    function build(){
      const cfg = api.load();

      modal.body.innerHTML = `
        <div class="trWrap">
          <div class="trTop">
            <div class="trPill">Default: <b id="trDef">${cfg.defaultKind}</b></div>
            <div class="trPill">Fade: <b id="trFade">${cfg.fadeMs}ms</b></div>
            <div class="trPill">Renderer frames: <b id="trFrames">0</b></div>
          </div>

          <button class="trTake" id="trTakeBtn" type="button">TAKE (PREVIEW → ONAIR)</button>

          <div class="trGrid">
            <div class="trCard">
              <div class="trRow">
                <div class="trLbl">Transition</div>
                <select class="trSel" id="trKind">
                  <option value="cut">cut</option>
                  <option value="fade">fade</option>
                  <option value="stinger">stinger</option>
                </select>

                <div class="trLbl">Fade ms</div>
                <input class="trInp" id="trFadeMs" placeholder="450" />

                <div class="trLbl">Stinger URL</div>
                <input class="trInp" id="trStinger" placeholder="optional (mp4/webm)" />
              </div>

              <div class="trBtns">
                <button class="trBtn primary" id="trRunBtn" type="button">RUN NOW (PREVIEW)</button>
                <button class="trBtn" id="trSaveBtn" type="button">SAVE DEFAULT</button>
              </div>

              <pre class="trPre" id="trPayload">{}</pre>
              <div class="trNote">
                This sends <b>postMessage</b> to your embedded program renderer iframe (preview only).
                Your renderer should interpret "TAKE" as a request to switch PROGRAM output (used by RTMP egress).
              </div>
            </div>
          </div>
        </div>
      `;

      const kindEl = document.getElementById("trKind");
      const fadeEl = document.getElementById("trFadeMs");
      const stingEl = document.getElementById("trStinger");
      kindEl.value = cfg.defaultKind || "cut";
      fadeEl.value = String(cfg.fadeMs || 450);
      stingEl.value = cfg.stingerUrl || "";

      function updatePayload(kind){
        const k = kind || kindEl.value;
        const fadeMs = parseInt(fadeEl.value, 10);
        const stingerUrl = stingEl.value || "";
        const payload = api.command(k, { fadeMs: Number.isFinite(fadeMs)?fadeMs:cfg.fadeMs, stingerUrl });
        document.getElementById("trPayload").textContent = JSON.stringify(payload, null, 2);
        document.getElementById("trDef").textContent = (api.load().defaultKind || "cut");
        document.getElementById("trFade").textContent = (api.load().fadeMs || 0) + "ms";
        const frames = findRendererFrames().length;
        document.getElementById("trFrames").textContent = String(frames);
        return payload;
      }

      function runNow(mode){
        const payload = updatePayload(kindEl.value);
        const frames = postToRenderer("cp:transition:run", Object.assign({ mode: mode || "preview" }, payload));
        CP.bus.emit("transition:run", { payload, mode: mode || "preview", frames });
        modal.setPill(frames ? "SENT" : "NO RENDERER");
      }

      async function take(){
        const payload = updatePayload(kindEl.value);
        const msg = Object.assign({ mode:"take" }, payload);
        const frames = postToRenderer("cp:transition:run", msg);
        CP.bus.emit("program:take", { payload: msg, frames });
        try{ await api.postTake(msg); }catch(_){}
        modal.setPill(frames ? "TAKE SENT" : "TAKE (NO RENDERER)");
      }

      function saveDefault(){
        api.save({ defaultKind: kindEl.value, fadeMs: parseInt(fadeEl.value,10) || 450, stingerUrl: stingEl.value || "" });
        updatePayload(kindEl.value);
        modal.setPill("SAVED");
      }

      document.getElementById("trRunBtn").addEventListener("click", ()=>runNow("preview"));
      document.getElementById("trTakeBtn").addEventListener("click", take);
      document.getElementById("trSaveBtn").addEventListener("click", saveDefault);

      kindEl.addEventListener("change", ()=>updatePayload(kindEl.value));
      fadeEl.addEventListener("input", ()=>updatePayload(kindEl.value));
      stingEl.addEventListener("input", ()=>updatePayload(kindEl.value));

      updatePayload(kindEl.value);
    }

    function open(){ build(); modal.open(); }

    const hook = CP.hookClick("#qaTake", open, "append");
    if (!hook.ok) CP.dom.addDockButton({ id:"cpDockTransitions", icon:"🎬", title:"Transitions / TAKE", onClick: open });

    window.CPTransitionsUI = { open };
  }

  if (window.CP && typeof window.CP.registerPlugin === "function"){
    window.CP.registerPlugin({ id:"transitions", init });
  } else {
    console.warn("[transitions] CP plugin host not found. Include addons/plugin_host.js first.");
  }
})();
