/* =========================================================
   RTMP Channels Plugin — logic (CP plugin-ready)
   File: addons/rtmp_channels.logic.js

   Stores channels in localStorage and starts/stops egress via backend.

   Suggested backend:
   POST /agent/egress/start  { room, requestedBy, rendererUrl, outputs:[{id,name,rtmpUrl}] }
   POST /agent/egress/stop   { room, ids:[id], jobIds:[jobId] }

   For UI testing without backend:
   add ?rtmpMock=1 to URL
========================================================= */
(function(){
  "use strict";

  const LS_CHANNELS = "rtmp_channels_v1";
  const LS_STATE = "rtmp_channels_state_v1";

  const safeParse = (s, fb)=>{ try{ return JSON.parse(s); }catch(_){ return fb; } };
  const lsGet = (k, fb)=>{ try{ const v = localStorage.getItem(k); return v==null?fb:v; }catch(_){ return fb; } };
  const lsSet = (k, v)=>{ try{ localStorage.setItem(k, v); }catch(_){ } };

  function qsGet(k, def=""){ try{ return (new URL(location.href)).searchParams.get(k) ?? def; }catch(_){ return def; } }
  function nowISO(){ return new Date().toISOString(); }

  function getRoom(){
    const q = qsGet("room","") || qsGet("channel","");
    if (q) return q;
    const lbl = document.getElementById("roomLbl");
    const t = (lbl && (lbl.textContent||"").trim()) || "";
    return (t && t !== "—") ? t : "";
  }

  function getRequester(){
    const hostLbl = document.getElementById("hostLbl");
    const h = (hostLbl && (hostLbl.textContent||"").trim()) || "";
    return h ? ("host-" + h) : "coproducer";
  }

  function defaultRendererUrl(){
    try{
      const u = new URL(location.origin + "/program_renderer.html");
      const room = getRoom();
      if (room) u.searchParams.set("room", room);
      u.searchParams.set("mode","program");
      return u.toString();
    }catch(_){ return ""; }
  }

  function endpoints(){
    const w = (window.RTMP_CHANNELS_ENDPOINTS && typeof window.RTMP_CHANNELS_ENDPOINTS === "object") ? window.RTMP_CHANNELS_ENDPOINTS : null;
    return {
      start: (w && w.start) || "/agent/egress/start",
      stop:  (w && w.stop)  || "/agent/egress/stop"
    };
  }

  function isMockMode(){
    if (window.RTMP_MOCK === true) return true;
    const q = qsGet("rtmpMock","0");
    if (q === "1" || q === "true") return true;
    const ls = lsGet("rtmp_mock","0");
    return (ls === "1" || ls === "true");
  }

  async function postJSON(url, body){
    const res = await fetch(url, {
      method:"POST",
      headers:{ "Content-Type":"application/json" },
      credentials:"same-origin",
      body: JSON.stringify(body || {})
    });
    const text = await res.text().catch(()=> "");
    const data = safeParse(text, null);
    if (!res.ok){
      const msg = (data && (data.error || data.message)) || text || (res.status + " " + res.statusText);
      const e = new Error(msg);
      e.status = res.status;
      throw e;
    }
    return data || {};
  }

  function loadChannels(){
    const raw = lsGet(LS_CHANNELS, "");
    const arr = safeParse(raw, null);
    if (Array.isArray(arr) && arr.length) return arr;
    return [
      { id:"youtube",  name:"YouTube",  rtmpUrl:"rtmp://a.rtmp.youtube.com/live2/XXXX-XXXX-XXXX-XXXX" },
      { id:"facebook", name:"Facebook", rtmpUrl:"rtmps://live-api-s.facebook.com:443/rtmp/XXXX?key=YYYY" },
      { id:"custom1",  name:"RTMP 1",   rtmpUrl:"rtmp://your.rtmp.server/app/streamKey" }
    ];
  }

  function saveChannels(list){
    const cleaned = (Array.isArray(list)?list:[]).map(c=>({
      id: String(c.id||"").trim(),
      name: String(c.name||"").trim(),
      rtmpUrl: String(c.rtmpUrl||"").trim()
    })).filter(c=>c.id && c.name && c.rtmpUrl);
    lsSet(LS_CHANNELS, JSON.stringify(cleaned));
    return cleaned;
  }

  function loadState(){ return safeParse(lsGet(LS_STATE,""), {}) || {}; }
  function saveState(st){ lsSet(LS_STATE, JSON.stringify(st||{})); }

  function maskUrl(u){
    if (!u) return "";
    try{
      const url = new URL(u);
      const tail = (url.pathname + url.search).slice(-10);
      return `${url.protocol}//${url.host}/…${tail}`;
    }catch(_){
      const s = String(u);
      if (s.length <= 18) return s;
      return s.slice(0, 12) + "…" + s.slice(-6);
    }
  }

  async function startOutputs(outputs, opts){
    const room = (opts && opts.room) || getRoom();
    if (!room) throw new Error("Room not set (missing ?room=...)");
    const payload = {
      room,
      requestedBy: ((opts && opts.requestedBy) || getRequester()),
      rendererUrl: ((opts && opts.rendererUrl) || defaultRendererUrl()),
      outputs: (outputs||[]).map(o=>({ id:o.id, name:o.name, rtmpUrl:o.rtmpUrl }))
    };

    if (isMockMode()){
      return { ok:true, mock:true, jobs: payload.outputs.map(o=>({ id:o.id, jobId:"MOCK_"+o.id+"_"+Date.now() })), payload };
    }
    return await postJSON(endpoints().start, payload);
  }

  async function stopOutputs(ids, jobIds, opts){
    const room = (opts && opts.room) || getRoom();
    if (!room) throw new Error("Room not set (missing ?room=...)");
    const payload = { room, ids: ids||[], jobIds: jobIds||[] };

    if (isMockMode()) return { ok:true, mock:true, payload };
    return await postJSON(endpoints().stop, payload);
  }

  const API = {
    getRoom, getRequester, defaultRendererUrl, endpoints, isMockMode,
    loadChannels, saveChannels, loadState, saveState, maskUrl,

    async startSelected(selectedIds, opts){
      const list = loadChannels();
      const outputs = list.filter(c => (selectedIds||[]).includes(c.id));
      if (!outputs.length) throw new Error("No channels selected.");
      const res = await startOutputs(outputs, opts);

      const st = loadState();
      const rk = (opts && opts.room) || getRoom();
      st[rk] = st[rk] || { jobs:{} };
      st[rk].jobs = st[rk].jobs || {};

      for (const j of (Array.isArray(res.jobs)?res.jobs:[])){
        if (j && j.id && j.jobId) st[rk].jobs[j.id] = { jobId:j.jobId, startedAt: nowISO() };
      }
      saveState(st);
      return res;
    },

    async stopSelected(selectedIds, opts){
      const rk = (opts && opts.room) || getRoom();
      const st = loadState();
      const jobs = (st[rk] && st[rk].jobs) ? st[rk].jobs : {};
      const jobIds = [];
      for (const id of (selectedIds||[])){
        if (jobs[id] && jobs[id].jobId) jobIds.push(jobs[id].jobId);
      }
      const res = await stopOutputs(selectedIds||[], jobIds, opts);
      if (st[rk] && st[rk].jobs){
        for (const id of (selectedIds||[])) delete st[rk].jobs[id];
      }
      saveState(st);
      return res;
    },

    isRunning(channelId, roomKey){
      const rk = roomKey || getRoom();
      const st = loadState();
      return !!(st[rk] && st[rk].jobs && st[rk].jobs[channelId] && st[rk].jobs[channelId].jobId);
    }
  };

  window.RTMPChannels = API;
})();
