/* =========================================================
   RTMP Channels Plugin — UI (CP plugin-ready)
   File: addons/rtmp_channels.ui.js

   - Hooks #btnStream and/or #qaStream if found
   - Otherwise adds a dock button (top-right)
========================================================= */
(function(){
  "use strict";

  function escapeHtml(s){
    return String(s).replace(/[&<>"']/g, (c)=>({ "&":"&amp;","<":"&lt;",">":"&gt;",'"':"&quot;","'":"&#39;" }[c]));
  }
  function tinyId(){ return "c_" + Math.random().toString(16).slice(2, 10) + "_" + Date.now().toString(16); }

  function ensureStyles(){
    if (document.getElementById("rtmpPluginStyles")) return;
    const css = `
      .rtmpBody{ padding:14px; }
      .rtmHeadRow{ display:flex; align-items:center; justify-content:space-between; gap:10px; margin:0 0 12px; flex-wrap:wrap; }
      .rtmRoomPill{ padding:6px 10px; border-radius:999px; border:1px solid rgba(255,255,255,.12);
        background:rgba(0,0,0,.18); font-size:12px; color:rgba(255,255,255,.88); display:inline-flex; gap:8px; align-items:center; }
      .rtmRoomPill b{ font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono","Courier New", monospace; font-weight:900; }

      .rtmList{ border:1px solid rgba(255,255,255,.10); border-radius:14px; overflow:hidden; background:rgba(0,0,0,.12); }
      .rtmRow2{ display:grid; grid-template-columns: 38px 1fr 140px 48px; gap:10px; align-items:center;
        padding:10px 12px; border-bottom:1px solid rgba(255,255,255,.08); }
      .rtmRow2:last-child{ border-bottom:none; }
      .rtmChk{ width:18px; height:18px; accent-color: rgba(59,130,246,.95); }
      .rtmName{ font-weight:1000; letter-spacing:.02em; }
      .rtmUrl{ font-size:11px; color:rgba(255,255,255,.72); overflow:hidden; text-overflow:ellipsis; white-space:nowrap; }
      .rtmLed{ width:14px; height:14px; border-radius:50%; border:1px solid rgba(255,255,255,.14);
        background: rgba(255,255,255,.12); box-shadow: inset 0 -2px 4px rgba(0,0,0,.55); }
      .rtmLed.on{ background:#22c55e; border-color: rgba(34,197,94,.65);
        box-shadow: 0 0 0 4px rgba(34,197,94,.10), 0 0 18px rgba(34,197,94,.35), inset 0 -2px 4px rgba(0,0,0,.35);
        animation: rtmpPulse 0.95s ease-in-out infinite; }
      @keyframes rtmpPulse{ 0%,100%{ filter:brightness(1); } 50%{ filter:brightness(1.4); } }
      .rtmMiniBtn{ height:30px; width:48px; border-radius:12px; border:1px solid rgba(255,255,255,.14);
        background:linear-gradient(180deg, rgba(255,255,255,.10), rgba(0,0,0,.22));
        color:rgba(255,255,255,.92); font-weight:1000; cursor:pointer; }
      .rtmMiniBtn:hover{ filter:brightness(1.08); }

      .rtmForm{ margin-top:12px; border:1px solid rgba(255,255,255,.10); border-radius:14px; background:rgba(0,0,0,.12); padding:12px; }
      .rtmFormHdr{ display:flex; justify-content:space-between; align-items:center; gap:10px; margin-bottom:10px; flex-wrap:wrap; }
      .rtmFormHdr b{ font-size:12px; letter-spacing:.10em; text-transform:uppercase; }
      .rtmHelpSmall{ font-size:11px; color:rgba(255,255,255,.62); }
      .rtmInlineGrid{ display:grid; grid-template-columns: 140px 1fr; gap:10px 12px; align-items:center; }
      .rtmInlineGrid input{ width:100%; padding:10px 10px; border-radius:12px; border:1px solid rgba(255,255,255,.12);
        background:rgba(0,0,0,.22); color:rgba(255,255,255,.92); outline:none; }
      .rtmInlineGrid input:focus{ border-color: rgba(59,130,246,.55); box-shadow: 0 0 0 4px rgba(59,130,246,.12); }
      .rtmLbl{ color:rgba(255,255,255,.72); font-size:12px; }
      .rtmBtns2{ display:flex; gap:10px; justify-content:flex-end; padding-top:10px; flex-wrap:wrap; }
      .rtmBtn{ border: 1px solid rgba(255,255,255,.14); background: rgba(255,255,255,.08); color: rgba(255,255,255,.92);
        padding: 10px 12px; border-radius: 12px; cursor: pointer; font-weight: 1000; }
      .rtmBtn:hover{ filter:brightness(1.08); }
      .rtmBtnOk{ border-color: rgba(34,197,94,.45); background: rgba(34,197,94,.12); }
      .rtmBtnDanger{ border-color: rgba(239,68,68,.45); background: rgba(239,68,68,.12); }
      .rtmDbg{ margin-top:10px; }
      .rtmDbg summary{ cursor:pointer; color:rgba(255,255,255,.76); font-weight:900; }
      .rtmPre{ margin:10px 0 0 0; padding:10px; border-radius:12px; border:1px solid rgba(255,255,255,.10);
        background:rgba(0,0,0,.22); max-height:220px; overflow:auto; font-size:12px; color: rgba(255,255,255,.86); }
    `;
    const style = document.createElement("style");
    style.id = "rtmpPluginStyles";
    style.textContent = css;
    document.head.appendChild(style);
  }

  function init(CP){
    if (!window.RTMPChannels){
      console.warn("[rtmp_channels] window.RTMPChannels missing (logic not loaded?)");
      return;
    }
    ensureStyles();

    const api = window.RTMPChannels;
    const modal = CP.dom.createModal({ id:"cp_rtmp_modal", title:"Stream Channels", pillText:"IDLE" });

    let selectedEditId = "";
    let lastSelectedIds = new Set();

    function logLine(msg, obj){
      const pre = document.getElementById("rtmLog");
      if (!pre) return;
      const line = `[${new Date().toLocaleTimeString()}] ${msg}` + (obj ? `\n${JSON.stringify(obj, null, 2)}` : "");
      pre.textContent = (pre.textContent ? (pre.textContent + "\n\n") : "") + line;
      pre.scrollTop = pre.scrollHeight;
    }

    function buildBody(){
      modal.body.innerHTML = `
        <div class="rtmpBody">
          <div class="rtmHeadRow">
            <div class="rtmRoomPill">Room: <b id="rtmRoomLbl">—</b></div>
            <div class="rtmBtns2">
              <button class="rtmBtn rtmBtnOk" id="rtmStartBtn" type="button">START SELECTED</button>
              <button class="rtmBtn rtmBtnDanger" id="rtmStopBtn" type="button">STOP SELECTED</button>
            </div>
          </div>

          <div class="rtmList" id="rtmList"></div>

          <div class="rtmForm">
            <div class="rtmFormHdr">
              <b>Add / Edit</b>
              <div class="rtmHelpSmall" id="rtmEditHint">Select a row to edit, or add a new channel.</div>
            </div>

            <div class="rtmInlineGrid">
              <div class="rtmLbl">Channel name</div>
              <input id="rtmNameInp" placeholder="e.g. YouTube" />

              <div class="rtmLbl">RTMP URL</div>
              <input id="rtmUrlInp" placeholder="rtmp://server/app/streamKey" />

              <div class="rtmLbl">Channel id</div>
              <input id="rtmIdInp" placeholder="auto (or e.g. youtube)" />

              <div class="rtmLbl"></div>
              <div class="rtmHelpSmall">Tip: URLs are masked in the list; click ✎ to edit.</div>
            </div>

            <div class="rtmBtns2">
              <button class="rtmBtn rtmBtnOk" id="rtmSaveBtn" type="button">SAVE</button>
              <button class="rtmBtn rtmBtnDanger" id="rtmDeleteBtn" type="button" disabled>DELETE</button>
            </div>

            <details class="rtmDbg">
              <summary>Debug</summary>
              <pre class="rtmPre" id="rtmLog"></pre>
            </details>
          </div>
        </div>
      `;
    }

    function loadChannels(){
      const list = api.loadChannels();
      api.saveChannels(list);
      return list;
    }
    function saveChannels(list){ return api.saveChannels(list); }
    function tryGetRoom(){ return api.getRoom ? api.getRoom() : ""; }
    function getRunning(id){ return api.isRunning(id, tryGetRoom()); }

    function setPill(text){ modal.setPill(text); }

    function pickForEdit(id){
      const list = loadChannels();
      const ch = list.find(x=>x.id===id);
      if (!ch) return;
      selectedEditId = ch.id;
      document.getElementById("rtmNameInp").value = ch.name || "";
      document.getElementById("rtmUrlInp").value = ch.rtmpUrl || "";
      document.getElementById("rtmIdInp").value = ch.id || "";
      document.getElementById("rtmDeleteBtn").disabled = false;
      const hint = document.getElementById("rtmEditHint");
      if (hint) hint.textContent = "Editing: " + (ch.name || ch.id);
    }

    function clearEdit(){
      selectedEditId = "";
      document.getElementById("rtmNameInp").value = "";
      document.getElementById("rtmUrlInp").value = "";
      document.getElementById("rtmIdInp").value = "";
      document.getElementById("rtmDeleteBtn").disabled = true;
      const hint = document.getElementById("rtmEditHint");
      if (hint) hint.textContent = "Select a row to edit, or add a new channel.";
    }

    function buildList(){
      const listEl = document.getElementById("rtmList");
      if (!listEl) return;

      const list = loadChannels();
      const room = tryGetRoom() || "—";
      document.getElementById("rtmRoomLbl").textContent = room;

      listEl.innerHTML = "";

      if (!list.length){
        const empty = document.createElement("div");
        empty.style.padding = "12px";
        empty.style.color = "rgba(255,255,255,.72)";
        empty.textContent = "No channels configured. Add one below.";
        listEl.appendChild(empty);
        return;
      }

      for (const ch of list){
        const row = document.createElement("div");
        row.className = "rtmRow2";
        row.dataset.id = ch.id;

        const running = getRunning(ch.id);
        const masked = api.maskUrl ? api.maskUrl(ch.rtmpUrl) : ch.rtmpUrl;

        row.innerHTML = `
          <input class="rtmChk" type="checkbox" data-chk="${escapeHtml(ch.id)}" ${lastSelectedIds.has(ch.id) ? "checked":""} />
          <div>
            <div class="rtmName">${escapeHtml(ch.name)}</div>
            <div class="rtmUrl">${escapeHtml(masked || "")}</div>
          </div>
          <div style="display:flex; align-items:center; gap:10px; justify-content:flex-end;">
            <div class="rtmLed ${running ? "on":""}" title="${running ? "RUNNING":"STOPPED"}"></div>
            <div class="rtmUrl" style="width:90px; text-align:right;">${running ? "LIVE":"OFF"}</div>
          </div>
          <button class="rtmMiniBtn" type="button" data-edit="${escapeHtml(ch.id)}" title="Edit">✎</button>
        `;
        listEl.appendChild(row);
      }

      listEl.querySelectorAll("input[data-chk]").forEach((chk)=>{
        chk.addEventListener("change", ()=>{
          const id = chk.getAttribute("data-chk");
          if (!id) return;
          if (chk.checked) lastSelectedIds.add(id);
          else lastSelectedIds.delete(id);
        });
      });

      listEl.querySelectorAll("button[data-edit]").forEach((btn)=>{
        btn.addEventListener("click", ()=>{
          const id = btn.getAttribute("data-edit") || "";
          pickForEdit(id);
        });
      });

      if (selectedEditId && !loadChannels().find(x=>x.id===selectedEditId)) clearEdit();
    }

    async function startSelected(){
      const ids = Array.from(lastSelectedIds);
      if (!ids.length){ alert("Select at least one channel."); return; }
      setPill("STARTING…"); logLine("startSelected", { room: tryGetRoom(), ids });

      try{ const res = await api.startSelected(ids, { room: tryGetRoom() }); setPill("STREAMING"); logLine("start ok", res); }
      catch(e){ setPill("ERROR"); logLine("start error", { message: String(e && e.message ? e.message : e) }); alert(String(e && e.message ? e.message : e)); }
      finally{ buildList(); }
    }

    async function stopSelected(){
      const ids = Array.from(lastSelectedIds);
      if (!ids.length){ alert("Select at least one channel."); return; }
      setPill("STOPPING…"); logLine("stopSelected", { room: tryGetRoom(), ids });

      try{ const res = await api.stopSelected(ids, { room: tryGetRoom() }); setPill("IDLE"); logLine("stop ok", res); }
      catch(e){ setPill("ERROR"); logLine("stop error", { message: String(e && e.message ? e.message : e) }); alert(String(e && e.message ? e.message : e)); }
      finally{ buildList(); }
    }

    function saveEdit(){
      const name = (document.getElementById("rtmNameInp").value || "").trim();
      const url  = (document.getElementById("rtmUrlInp").value || "").trim();
      let id = (document.getElementById("rtmIdInp").value || "").trim();

      if (!name || !url){ alert("Name and RTMP URL are required."); return; }
      if (!id) id = selectedEditId || tinyId();

      let list = loadChannels();
      const existing = list.find(x=>x.id===id);
      if (existing){ existing.name = name; existing.rtmpUrl = url; }
      else list.push({ id, name, rtmpUrl:url });

      list = saveChannels(list);
      logLine("saved channel", { id, name });
      selectedEditId = id;
      buildList();
      pickForEdit(id);
    }

    function deleteEdit(){
      if (!selectedEditId) return;
      if (!confirm("Delete this channel?")) return;

      let list = loadChannels();
      list = list.filter(x=>x.id!==selectedEditId);
      saveChannels(list);

      lastSelectedIds.delete(selectedEditId);
      logLine("deleted channel", { id: selectedEditId });

      clearEdit();
      buildList();
    }

    function open(){
      buildBody();
      buildList();
      const list = loadChannels();
      setPill(list.some(ch => getRunning(ch.id)) ? "STREAMING" : "IDLE");

      document.getElementById("rtmStartBtn").addEventListener("click", startSelected);
      document.getElementById("rtmStopBtn").addEventListener("click", stopSelected);
      document.getElementById("rtmSaveBtn").addEventListener("click", saveEdit);
      document.getElementById("rtmDeleteBtn").addEventListener("click", deleteEdit);

      modal.open();
    }

    const hookedTop = CP.hookClick("#btnStream", open, "replace");
    const hookedQA  = CP.hookClick("#qaStream", open, "replace");
    if (!hookedTop.ok && !hookedQA.ok){
      CP.dom.addDockButton({ id:"cpDockStream", icon:"⏺", title:"Stream Channels", onClick: open });
    }

    window.RTMPChannelsUI = { open };
    buildBody(); buildList();
  }

  if (window.CP && typeof window.CP.registerPlugin === "function"){
    window.CP.registerPlugin({ id:"rtmp_channels", init });
  } else {
    console.warn("[rtmp_channels] CP plugin host not found. Include addons/plugin_host.js first.");
  }
})();
