/* =========================================================
   Co-Producer Addons — CORE BRIDGE (LiveKit room capture)
   - Captures the LiveKit Room instance without touching main page
   - Exposes: window.CP_CTX.getRoom(), window.CP_CTX.getLK()
   - Emits: CP.bus 'lk:room' when a Room is constructed
========================================================= */
(function(){
  "use strict";

  function getLK(){
    return window.LivekitClient || window.LK || window.livekitClient || window.livekit || null;
  }

  function ensureCtx(){
    if (!window.CP_CTX) window.CP_CTX = {};
    window.CP_CTX.getLK = getLK;
    window.CP_CTX.getRoom = () => window.__CP_LK_ROOM__ || null;
    window.CP_CTX.waitForRoom = async (timeoutMs=10000) => {
      const t0 = Date.now();
      while (Date.now() - t0 < timeoutMs){
        const r = window.CP_CTX.getRoom();
        if (r) return r;
        await new Promise(res => setTimeout(res, 80));
      }
      return null;
    };
  }

  function wrapRoom(){
    const LK = getLK();
    if (!LK || !LK.Room) return false;
    if (LK.Room && LK.Room.__cpWrapped) return true;

    const OrigRoom = LK.Room;

    class CPWrappedRoom extends OrigRoom{
      constructor(...args){
        super(...args);
        try{
          window.__CP_LK_ROOM__ = this;
          ensureCtx();
          if (window.CP && window.CP.bus && typeof window.CP.bus.emit === "function"){
            window.CP.bus.emit("lk:room", { room: this });
          }
        }catch(_){}
      }
    }

    // Copy static properties (best-effort)
    try{
      for (const k of Object.getOwnPropertyNames(OrigRoom)){
        if (k === "prototype") continue;
        const d = Object.getOwnPropertyDescriptor(OrigRoom, k);
        if (!d) continue;
        try{ Object.defineProperty(CPWrappedRoom, k, d); }catch(_){}
      }
    }catch(_){}

    CPWrappedRoom.__cpWrapped = true;
    LK.Room = CPWrappedRoom;

    ensureCtx();
    return true;
  }

  function install(){
    ensureCtx();
    if (wrapRoom()) return;

    // Retry a bit later if LK SDK loads after us
    let tries = 0;
    const iv = setInterval(() => {
      tries++;
      if (wrapRoom() || tries > 80){
        clearInterval(iv);
      }
    }, 120);
  }

  // Register as a plugin (no UI)
  if (window.CP && typeof window.CP.registerPlugin === "function"){
    window.CP.registerPlugin({
      id: "core_bridge",
      name: "Core Bridge",
      init(){ install(); }
    });
  } else {
    // If plugin host isn't ready yet, still install (safe)
    install();
  }
})();
